<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Utility;

use Drupal\search_api\IndexInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;

/**
 * Helper class for managing index-level settings.
 *
 * Provides methods to get default settings and merge them with
 * index third-party settings for the search_api_sqlite module.
 */
final class IndexSettings {

  /**
   * The module name for third-party settings.
   */
  public const string MODULE_NAME = 'search_api_sqlite';

  /**
   * Gets the default index settings.
   *
   * @return array<string, mixed>
   *   The default settings array.
   */
  public static function defaults(): array {
    return [
      'tokenizer' => Tokenizer::Unicode61->value,
      'trigram_case_sensitive' => FALSE,
      'min_chars' => 3,
      'matching' => MatchingMode::Words->value,
      'highlighting' => [
        'enabled' => TRUE,
        'prefix' => '<strong>',
        'suffix' => '</strong>',
        'excerpt_length' => 256,
      ],
      'optimization' => [
        'auto_optimize' => TRUE,
        'optimize_threshold' => 1000,
      ],
    ];
  }

  /**
   * Merges provided settings with defaults.
   *
   * @param array<string, mixed> $settings
   *   The settings to merge with defaults.
   *
   * @return array<string, mixed>
   *   The merged settings array.
   */
  public static function mergeDefaults(array $settings): array {
    $defaults = self::defaults();

    // Deep merge for nested arrays.
    foreach ($defaults as $key => $default_value) {
      if (!isset($settings[$key])) {
        $settings[$key] = $default_value;
      }
      elseif (is_array($default_value) && is_array($settings[$key])) {
        $settings[$key] = array_merge($default_value, $settings[$key]);
      }
    }

    return $settings;
  }

  /**
   * Gets the settings for a specific index.
   *
   * Retrieves third-party settings from the index and merges with defaults.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   *
   * @return array<string, mixed>
   *   The merged settings for the index.
   */
  public static function getIndexSettings(IndexInterface $index): array {
    $settings = $index->getThirdPartySettings(self::MODULE_NAME);
    return self::mergeDefaults($settings);
  }

  /**
   * Gets a specific setting value for an index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param string $key
   *   The setting key.
   * @param mixed $default
   *   The default value if setting is not found.
   *
   * @return mixed
   *   The setting value.
   */
  public static function getIndexSetting(IndexInterface $index, string $key, mixed $default = NULL): mixed {
    $settings = self::getIndexSettings($index);
    return $settings[$key] ?? $default;
  }

  /**
   * Gets nested setting value using dot notation.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param string $key
   *   The setting key in dot notation (e.g., 'highlighting.enabled').
   * @param mixed $default
   *   The default value if setting is not found.
   *
   * @return mixed
   *   The setting value.
   */
  public static function getNestedSetting(IndexInterface $index, string $key, mixed $default = NULL): mixed {
    $settings = self::getIndexSettings($index);
    $keys = explode('.', $key);

    $value = $settings;
    foreach ($keys as $nested_key) {
      if (!is_array($value) || !array_key_exists($nested_key, $value)) {
        return $default;
      }

      $value = $value[$nested_key];
    }

    return $value;
  }

}
