<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Drupal\search_api\Entity\Index;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;
use Drupal\search_api_sqlite\Utility\IndexSettings;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests index third-party settings functionality.
 */
#[CoversClass(IndexSettings::class)]
#[Group('search_api_sqlite')]
final class IndexThirdPartySettingsTest extends SqliteFts5TestBase {

  /**
   * Tests that third-party settings are saved and loaded correctly.
   */
  public function testSettingsAreSavedAndLoaded(): void {
    // Set custom third-party settings.
    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'tokenizer', Tokenizer::Porter->value);
    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'min_chars', 5);
    $this->index->save();

    // Reload the index.
    $reloaded_index = Index::load($this->index->id());
    $this->assertNotNull($reloaded_index);

    // Verify settings persisted.
    $tokenizer = $reloaded_index->getThirdPartySetting(IndexSettings::MODULE_NAME, 'tokenizer');
    $min_chars = $reloaded_index->getThirdPartySetting(IndexSettings::MODULE_NAME, 'min_chars');

    $this->assertEquals(Tokenizer::Porter->value, $tokenizer);
    $this->assertEquals(5, $min_chars);
  }

  /**
   * Tests that IndexSettings::getIndexSettings() works with stored settings.
   */
  public function testGetIndexSettingsWithStoredSettings(): void {
    // Set partial settings.
    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'tokenizer', Tokenizer::Trigram->value);
    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'trigram_case_sensitive', TRUE);
    $this->index->save();

    // Reload and get merged settings.
    $reloaded_index = Index::load($this->index->id());
    $settings = IndexSettings::getIndexSettings($reloaded_index);

    // Stored values preserved.
    $this->assertEquals(Tokenizer::Trigram->value, $settings['tokenizer']);
    $this->assertTrue($settings['trigram_case_sensitive']);

    // Defaults filled in.
    $this->assertEquals(3, $settings['min_chars']);
    $this->assertEquals(MatchingMode::Words->value, $settings['matching']);
    $this->assertArrayHasKey('highlighting', $settings);
    $this->assertTrue($settings['highlighting']['enabled']);
  }

  /**
   * Tests that nested settings are saved correctly.
   */
  public function testNestedSettingsAreSaved(): void {
    $highlighting_settings = [
      'enabled' => FALSE,
      'prefix' => '<mark>',
      'suffix' => '</mark>',
      'excerpt_length' => 128,
    ];

    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'highlighting', $highlighting_settings);
    $this->index->save();

    // Reload the index.
    $reloaded_index = Index::load($this->index->id());
    $stored_highlighting = $reloaded_index->getThirdPartySetting(IndexSettings::MODULE_NAME, 'highlighting');

    $this->assertFalse($stored_highlighting['enabled']);
    $this->assertEquals('<mark>', $stored_highlighting['prefix']);
    $this->assertEquals('</mark>', $stored_highlighting['suffix']);
    $this->assertEquals(128, $stored_highlighting['excerpt_length']);
  }

  /**
   * Tests that the backend correctly uses index settings.
   */
  public function testBackendUsesIndexSettings(): void {
    // Set custom tokenizer on the index.
    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'tokenizer', Tokenizer::Porter->value);
    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'matching', MatchingMode::Prefix->value);
    $this->index->save();

    // Get merged config from backend.
    $config = $this->backend->getIndexConfiguration($this->index);

    // Index settings should override defaults.
    $this->assertEquals(Tokenizer::Porter->value, $config['tokenizer']);
    $this->assertEquals(MatchingMode::Prefix->value, $config['matching']);

    // Backend settings should still be present.
    $this->assertArrayHasKey('concurrency', $config);
    $this->assertArrayHasKey('verbose_logging', $config);
  }

  /**
   * Tests that index settings are independent between indexes.
   */
  public function testIndexSettingsAreIndependent(): void {
    // Create a second index.
    $second_index = Index::create([
      'id' => 'test_index_2',
      'name' => 'Test Index 2',
      'server' => 'test_server',
      'datasource_settings' => [
        'entity:node' => [],
      ],
    ]);
    $second_index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'tokenizer', Tokenizer::Trigram->value);
    $second_index->save();

    // Set different tokenizer on first index.
    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'tokenizer', Tokenizer::Porter->value);
    $this->index->save();

    // Reload both indexes.
    $reloaded_first = Index::load($this->index->id());
    $reloaded_second = Index::load('test_index_2');

    // Verify they have different settings.
    $first_settings = IndexSettings::getIndexSettings($reloaded_first);
    $second_settings = IndexSettings::getIndexSettings($reloaded_second);

    $this->assertEquals(Tokenizer::Porter->value, $first_settings['tokenizer']);
    $this->assertEquals(Tokenizer::Trigram->value, $second_settings['tokenizer']);

    // Clean up second index.
    $second_index->delete();
  }

  /**
   * Tests that settings survive index update.
   */
  public function testSettingsSurviveIndexUpdate(): void {
    // Set custom settings.
    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'tokenizer', Tokenizer::Ascii->value);
    $this->index->setThirdPartySetting(IndexSettings::MODULE_NAME, 'min_chars', 2);
    $this->index->save();

    // Update the index (change name).
    $reloaded_index = Index::load($this->index->id());
    $reloaded_index->set('name', 'Updated Test Index');
    $reloaded_index->save();

    // Reload again and verify settings persist.
    $final_index = Index::load($this->index->id());
    $settings = IndexSettings::getIndexSettings($final_index);

    $this->assertEquals(Tokenizer::Ascii->value, $settings['tokenizer']);
    $this->assertEquals(2, $settings['min_chars']);
  }

}
