<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Entity\Server;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\ServerInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;
use Drupal\search_api_sqlite\Plugin\search_api\backend\SqliteFts5;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Base class for SQLite FTS5 kernel tests.
 *
 * Provides shared setup, teardown, and helper methods for all backend tests.
 */
abstract class SqliteFts5TestBase extends KernelTestBase {

  /**
   * {@inheritdoc}
   *
   * @var array<int, string>
   */
  protected static $modules = [
    'user',
    'file',
    'field',
    'text',
    'filter',
    'node',
    'system',
    'search_api',
    'search_api_sqlite',
  ];

  /**
   * The test directory path.
   */
  protected string $testDir;

  /**
   * The search server.
   */
  protected ServerInterface $server;

  /**
   * The search index.
   */
  protected IndexInterface $index;

  /**
   * The backend plugin.
   */
  protected SqliteFts5 $backend;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('search_api_server');
    $this->installEntitySchema('search_api_index');
    $this->installEntitySchema('search_api_task');
    $this->installSchema('search_api', ['search_api_item']);
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['filter', 'node', 'search_api']);

    // Set up test directory.
    $this->testDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($this->testDir, 0777, TRUE);
    mkdir($this->testDir . '/private', 0777, TRUE);

    // Configure private file path in settings.
    $this->setSetting('file_private_path', $this->testDir . '/private');

    // Install the module configuration.
    $this->installConfig(['search_api_sqlite']);

    // Update the module config to use our test directory.
    $config = $this->config('search_api_sqlite.settings');
    $config->set('database_path', $this->testDir . '/databases');
    $config->save();

    // Create database directory.
    mkdir($this->testDir . '/databases', 0777, TRUE);

    // Create server and index.
    $this->createTestServer();
    $this->createTestIndex();
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function tearDown(): void {
    // Clean up the database files and test directory.
    try {
      $this->backend->removeIndex($this->index);
    }
    catch (\Exception) {
      // Ignore cleanup errors.
    }

    // Remove test directory.
    if (is_dir($this->testDir)) {
      $this->recursiveDelete($this->testDir);
    }

    parent::tearDown();
  }

  /**
   * Creates the test server.
   */
  protected function createTestServer(): void {
    $this->server = Server::create([
      'id' => 'test_server',
      'name' => 'Test Server',
      'backend' => 'search_api_sqlite',
      'backend_config' => $this->getServerConfiguration(),
    ]);
    $this->server->save();
  }

  /**
   * Gets the server configuration.
   *
   * Can be overridden in subclasses for different configurations.
   *
   * @return array<string, mixed>
   *   The server backend configuration.
   */
  protected function getServerConfiguration(): array {
    return [
      'database_path' => $this->testDir . '/databases',
      'verbose_logging' => TRUE,
      'log_queries' => TRUE,
      'concurrency' => [
        'max_retries' => 3,
        'retry_delay' => 100,
        'busy_timeout' => 5000,
      ],
    ];
  }

  /**
   * Gets the index configuration (third-party settings).
   *
   * Can be overridden in subclasses for different configurations.
   *
   * @return array<string, mixed>
   *   The index third-party settings.
   */
  protected function getIndexConfiguration(): array {
    return [
      'tokenizer' => Tokenizer::Unicode61->value,
      'trigram_case_sensitive' => FALSE,
      'min_chars' => 3,
      'matching' => MatchingMode::Words->value,
      'highlighting' => [
        'enabled' => TRUE,
        'prefix' => '<em>',
        'suffix' => '</em>',
        'excerpt_length' => 256,
      ],
      'optimization' => [
        'auto_optimize' => TRUE,
        'optimize_threshold' => 1000,
      ],
    ];
  }

  /**
   * Creates the test index with standard fields.
   */
  protected function createTestIndex(): void {
    // Create an index.
    $this->index = Index::create([
      'id' => 'test_index',
      'name' => 'Test Index',
      'server' => 'test_server',
      'status' => TRUE,
      'tracker_settings' => [
        'default' => [],
      ],
      'datasource_settings' => [
        'entity:node' => [],
      ],
      'third_party_settings' => [
        'search_api_sqlite' => $this->getIndexConfiguration(),
      ],
    ]);

    // Add fields to the index.
    $fields_helper = \Drupal::service('search_api.fields_helper');

    $title_field = $fields_helper->createField($this->index, 'title', [
      'label' => 'Title',
      'type' => 'text',
      'datasource_id' => 'entity:node',
      'property_path' => 'title',
    ]);
    $this->index->addField($title_field);

    $nid_field = $fields_helper->createField($this->index, 'nid', [
      'label' => 'Node ID',
      'type' => 'integer',
      'datasource_id' => 'entity:node',
      'property_path' => 'nid',
    ]);
    $this->index->addField($nid_field);

    $status_field = $fields_helper->createField($this->index, 'status', [
      'label' => 'Status',
      'type' => 'boolean',
      'datasource_id' => 'entity:node',
      'property_path' => 'status',
    ]);
    $this->index->addField($status_field);

    $langcode_field = $fields_helper->createField($this->index, 'langcode', [
      'label' => 'Language',
      'type' => 'string',
      'datasource_id' => 'entity:node',
      'property_path' => 'langcode',
    ]);
    $this->index->addField($langcode_field);

    $body_field = $fields_helper->createField($this->index, 'body', [
      'label' => 'Body',
      'type' => 'text',
      'datasource_id' => 'entity:node',
      'property_path' => 'body',
    ]);
    $this->index->addField($body_field);

    $category_field = $fields_helper->createField($this->index, 'category', [
      'label' => 'Category',
      'type' => 'string',
      'datasource_id' => 'entity:node',
      'property_path' => 'field_category',
    ]);
    $this->index->addField($category_field);

    // Save the index.
    $this->index->save();

    // Reload the index to get the saved state.
    $this->index = Index::load('test_index');

    // Re-add body and category fields if they were dropped.
    $this->ensureFieldsExist($fields_helper);

    // Get the backend.
    $backend = $this->server->getBackend();
    assert($backend instanceof SqliteFts5);
    $this->backend = $backend;
    $this->backend->updateIndex($this->index);
  }

  /**
   * Ensures required fields exist on the index.
   *
   * @param mixed $fields_helper
   *   The fields helper service.
   */
  protected function ensureFieldsExist(mixed $fields_helper): void {
    $fields = $this->index->getFields();

    if (!isset($fields['body'])) {
      $body_field = $fields_helper->createField($this->index, 'body', [
        'label' => 'Body',
        'type' => 'text',
        'datasource_id' => 'entity:node',
        'property_path' => 'body',
      ]);
      $this->index->addField($body_field);
    }

    if (!isset($fields['category'])) {
      $category_field = $fields_helper->createField($this->index, 'category', [
        'label' => 'Category',
        'type' => 'string',
        'datasource_id' => 'entity:node',
        'property_path' => 'field_category',
      ]);
      $this->index->addField($category_field);
    }
  }

  /**
   * Recursively deletes a directory.
   *
   * @param string $dir
   *   Directory path.
   */
  protected function recursiveDelete(string $dir): void {
    if (!is_dir($dir)) {
      return;
    }

    $files = array_diff(scandir($dir) ?: [], ['.', '..']);
    foreach ($files as $file) {
      $path = $dir . '/' . $file;
      is_dir($path) ? $this->recursiveDelete($path) : unlink($path);
    }

    rmdir($dir);
  }

  /**
   * Creates a mock item for indexing.
   *
   * @param string $item_id
   *   The item ID.
   * @param array<string, mixed> $field_values
   *   The field values keyed by field name.
   *
   * @return \Drupal\search_api\Item\ItemInterface&\PHPUnit\Framework\MockObject\MockObject
   *   The mock item.
   */
  protected function createMockItem(string $item_id, array $field_values): ItemInterface&MockObject {
    $item = $this->createMock(ItemInterface::class);
    $item->method('getId')->willReturn($item_id);
    $item->method('getDatasourceId')->willReturn('entity:node');
    $item->method('getLanguage')->willReturn('en');

    $fields = [];
    foreach ($field_values as $field_name => $value) {
      $field = $this->createMock(FieldInterface::class);
      $field->method('getFieldIdentifier')->willReturn($field_name);
      $field->method('getValues')->willReturn(is_array($value) ? $value : [$value]);

      $index_field = $this->index->getField($field_name);
      $type = $index_field?->getType() ?? 'string';
      $field->method('getType')->willReturn($type);

      $fields[$field_name] = $field;
    }

    $item->method('getFields')->willReturn($fields);
    $item->method('getField')
      ->willReturnCallback(fn($field_name) => $fields[$field_name] ?? NULL);

    return $item;
  }

  /**
   * Indexes test items.
   *
   * @param array<array<string, mixed>> $items_data
   *   Array of item data, each with 'id' and field values.
   *
   * @return array<string>
   *   Array of indexed item IDs.
   */
  protected function indexTestItems(array $items_data): array {
    $items = [];
    foreach ($items_data as $data) {
      $item_id = $data['id'];
      unset($data['id']);
      $items[$item_id] = $this->createMockItem($item_id, $data);
    }

    return $this->backend->indexItems($this->index, $items);
  }

}
