<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Autocomplete;

use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_autocomplete\SearchInterface;
use Drupal\search_api_sqlite\Autocomplete\AutocompleteHandler;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Search\QueryBuilderInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;

/**
 * Tests the AutocompleteHandler service.
 */
#[CoversClass(AutocompleteHandler::class)]
#[Group('search_api_sqlite')]
final class AutocompleteHandlerTest extends UnitTestCase {

  /**
   * The autocomplete handler under test.
   */
  private AutocompleteHandler $handler;

  /**
   * The mocked connection manager.
   */
  private ConnectionManagerInterface $connectionManager;

  /**
   * The mocked schema manager.
   */
  private SchemaManagerInterface $schemaManager;

  /**
   * The mocked FTS5 query runner.
   */
  private Fts5QueryRunnerInterface $fts5QueryRunner;

  /**
   * The mocked query builder.
   */
  private QueryBuilderInterface $queryBuilder;

  /**
   * The mocked logger.
   */
  private LoggerInterface $logger;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->fts5QueryRunner = $this->createMock(Fts5QueryRunnerInterface::class);
    $this->queryBuilder = $this->createMock(QueryBuilderInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->handler = new AutocompleteHandler(
      $this->connectionManager,
      $this->schemaManager,
      $this->fts5QueryRunner,
      $this->queryBuilder,
      $this->logger,
    );
  }

  /**
   * Creates a mock query with index.
   *
   * @param string $index_id
   *   The index ID.
   *
   * @return \Drupal\search_api\Query\QueryInterface
   *   The mocked query.
   */
  private function createMockQuery(string $index_id = 'test_index'): QueryInterface {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn($index_id);

    $query = $this->createMock(QueryInterface::class);
    $query->method('getIndex')->willReturn($index);

    return $query;
  }

  /**
   * Creates a mock search configuration.
   *
   * @param int|null $limit
   *   The suggestion limit.
   *
   * @return \Drupal\search_api_autocomplete\SearchInterface
   *   The mocked search.
   */
  private function createMockSearch(?int $limit = 10): SearchInterface {
    $search = $this->createMock(SearchInterface::class);
    $search->method('getOption')->with('limit')->willReturn($limit);

    return $search;
  }

  /**
   * Tests getSuggestions returns empty when tables don't exist.
   */
  public function testGetSuggestionsReturnsEmptyWhenTablesDoNotExist(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')
      ->with('test_index')
      ->willReturn(FALSE);

    $result = $this->handler->getSuggestions(
      $query,
      $search,
      'dru',
      'dru',
      ['title' => 'title']
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getSuggestions returns empty for empty incomplete key.
   */
  public function testGetSuggestionsReturnsEmptyForEmptyIncompleteKey(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')->willReturn('fts_test_index');

    $result = $this->handler->getSuggestions(
      $query,
      $search,
      '',
      '',
      ['title' => 'title']
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getSuggestions returns empty for empty fulltext fields.
   */
  public function testGetSuggestionsReturnsEmptyForEmptyFulltextFields(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')->willReturn('fts_test_index');

    $result = $this->handler->getSuggestions(
      $query,
      $search,
      'dru',
      'dru',
      []
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getSuggestions returns empty when no FTS results found.
   */
  public function testGetSuggestionsReturnsEmptyWhenNoFtsResults(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')->willReturn('fts_test_index');

    $this->queryBuilder->method('escapeTerm')
      ->with('dru')
      ->willReturn('dru');

    $this->fts5QueryRunner->method('search')
      ->willReturn([]);

    $result = $this->handler->getSuggestions(
      $query,
      $search,
      'dru',
      'dru',
      ['title' => 'title']
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getSuggestions builds correct prefix query.
   */
  public function testGetSuggestionsBuildsPrefixQuery(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch(5);

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')->willReturn('fts_test_index');

    $this->queryBuilder->method('escapeTerm')
      ->with('test')
      ->willReturn('test');

    // Verify the prefix query format.
    $this->fts5QueryRunner->expects($this->once())
      ->method('search')
      ->with(
        'test_index',
        'fts_test_index',
        '"test"*',
        [
          'limit' => 5,
          'offset' => 0,
          'order_by_rank' => TRUE,
        ]
      )
      ->willReturn([]);

    $this->handler->getSuggestions(
      $query,
      $search,
      'test',
      'test',
      ['title' => 'title']
    );
  }

  /**
   * Tests getSuggestions escapes special characters in incomplete key.
   */
  public function testGetSuggestionsEscapesSpecialCharacters(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')->willReturn('fts_test_index');

    $this->queryBuilder->method('escapeTerm')
      ->with('test"key')
      ->willReturn('"test""key"');

    $this->fts5QueryRunner->expects($this->once())
      ->method('search')
      ->with(
        'test_index',
        'fts_test_index',
        '""test""key""*',
        $this->anything()
      )
      ->willReturn([]);

    $this->handler->getSuggestions(
      $query,
      $search,
      'test"key',
      'test"key',
      ['title' => 'title']
    );
  }

  /**
   * Tests getSuggestions logs errors on exception.
   */
  public function testGetSuggestionsLogsErrorsOnException(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')
      ->willThrowException(new \Exception('Database error'));

    $this->logger->expects($this->once())
      ->method('error')
      ->with(
        'Autocomplete failed for index @index: @error',
        $this->callback(fn($context): bool => $context['@index'] === 'test_index'
          && $context['@error'] === 'Database error')
      );

    $result = $this->handler->getSuggestions(
      $query,
      $search,
      'test',
      'test',
      ['title' => 'title']
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getSuggestions uses default limit when not specified.
   */
  public function testGetSuggestionsUsesDefaultLimit(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch(NULL);

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')->willReturn('fts_test_index');
    $this->queryBuilder->method('escapeTerm')->willReturn('test');

    $this->fts5QueryRunner->expects($this->once())
      ->method('search')
      ->with(
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->callback(fn($options): bool => $options['limit'] === 10)
      )
      ->willReturn([]);

    $this->handler->getSuggestions(
      $query,
      $search,
      'test',
      'test',
      ['title' => 'title']
    );
  }

  /**
   * Tests getSuggestions skips duplicate item IDs.
   */
  public function testGetSuggestionsSkipsDuplicateItemIds(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')->willReturn('fts_test_index');
    $this->queryBuilder->method('escapeTerm')->willReturn('test');

    // Return duplicate item IDs.
    $this->fts5QueryRunner->method('search')
      ->willReturn([
        ['item_id' => 'entity:node/1', 'rank' => 1.0],
        ['item_id' => 'entity:node/1', 'rank' => 0.9],
        ['item_id' => 'entity:node/2', 'rank' => 0.8],
      ]);

    // Mock PDO for extractCompletion.
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);
    $stmt->method('fetchColumn')->willReturn('testing content');

    $pdo = $this->createMock(\PDO::class);
    $pdo->method('prepare')->willReturn($stmt);

    $this->connectionManager->method('getPdo')->willReturn($pdo);

    // Note: Can't fully test suggestion creation without SuggestionFactory
    // being available, but we can verify the method doesn't error.
    $result = $this->handler->getSuggestions(
      $query,
      $search,
      'test',
      'test',
      ['title' => 'title']
    );

    // Result depends on SuggestionFactory availability.
    $this->assertIsArray($result);
  }

}
