<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Index;

use PHPUnit\Framework\MockObject\MockObject;
use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api\Utility\DataTypeHelperInterface;
use Drupal\search_api_sqlite\Index\FieldTypeMapper;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the FieldTypeMapper service.
 */
#[CoversClass(FieldTypeMapper::class)]
#[Group('search_api_sqlite')]
final class FieldTypeMapperTest extends UnitTestCase {

  /**
   * The field type mapper under test.
   */
  private FieldTypeMapper $mapper;

  /**
   * The mocked data type helper.
   */
  private MockObject $dataTypeHelper;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->dataTypeHelper = $this->createMock(DataTypeHelperInterface::class);
    $this->dataTypeHelper->method('isTextType')
      ->willReturnCallback(fn(string $type): bool => $type === 'text');

    $this->mapper = new FieldTypeMapper($this->dataTypeHelper);
  }

  /**
   * Tests isFulltextType() with various types.
   */
  #[DataProvider('fulltextTypeProvider')]
  public function testIsFulltextType(string $type, bool $expected): void {
    $this->assertSame($expected, $this->mapper->isFulltextType($type));
  }

  /**
   * Data provider for fulltext type tests.
   *
   * @return array<array{string, bool}>
   *   Test cases.
   */
  public static function fulltextTypeProvider(): array {
    return [
      'text is fulltext' => ['text', TRUE],
      'string is not fulltext' => ['string', FALSE],
      'integer is not fulltext' => ['integer', FALSE],
      'date is not fulltext' => ['date', FALSE],
      'boolean is not fulltext' => ['boolean', FALSE],
      'decimal is not fulltext' => ['decimal', FALSE],
      'uri is not fulltext' => ['uri', FALSE],
    ];
  }

  /**
   * Tests getStorageType() with various types.
   */
  #[DataProvider('storageTypeProvider')]
  public function testGetStorageType(string $type, string $expected): void {
    $this->assertSame($expected, $this->mapper->getStorageType($type));
  }

  /**
   * Data provider for storage type tests.
   *
   * @return array<array{string, string}>
   *   Test cases.
   */
  public static function storageTypeProvider(): array {
    return [
      'text maps to fts5' => ['text', FieldTypeMapperInterface::STORAGE_FTS5],
      'string maps to string' => ['string', FieldTypeMapperInterface::STORAGE_STRING],
      'uri maps to string' => ['uri', FieldTypeMapperInterface::STORAGE_STRING],
      'integer maps to integer' => ['integer', FieldTypeMapperInterface::STORAGE_INTEGER],
      'boolean maps to integer' => ['boolean', FieldTypeMapperInterface::STORAGE_INTEGER],
      'date maps to integer' => ['date', FieldTypeMapperInterface::STORAGE_INTEGER],
      'decimal maps to decimal' => ['decimal', FieldTypeMapperInterface::STORAGE_DECIMAL],
      'unknown maps to string' => ['unknown_type', FieldTypeMapperInterface::STORAGE_STRING],
    ];
  }

  /**
   * Tests getFieldDataColumn() with various storage types.
   */
  #[DataProvider('fieldDataColumnProvider')]
  public function testGetFieldDataColumn(string $storage_type, string $expected): void {
    $this->assertSame($expected, $this->mapper->getFieldDataColumn($storage_type));
  }

  /**
   * Data provider for field data column tests.
   *
   * @return array<array{string, string}>
   *   Test cases.
   */
  public static function fieldDataColumnProvider(): array {
    return [
      'string storage' => [FieldTypeMapperInterface::STORAGE_STRING, 'value_string'],
      'integer storage' => [FieldTypeMapperInterface::STORAGE_INTEGER, 'value_integer'],
      'decimal storage' => [FieldTypeMapperInterface::STORAGE_DECIMAL, 'value_decimal'],
    ];
  }

  /**
   * Tests getFieldDataColumn() with unknown storage type.
   */
  public function testGetFieldDataColumnWithUnknownType(): void {
    $this->assertSame('value_string', $this->mapper->getFieldDataColumn('unknown'));
  }

  /**
   * Tests convertValue() for text type.
   */
  public function testConvertValueText(): void {
    $this->assertSame('hello world', $this->mapper->convertValue('hello world', 'text'));
    $this->assertSame('stripped tags', $this->mapper->convertValue('<p>stripped tags</p>', 'text'));
    $this->assertSame('normalized spaces', $this->mapper->convertValue("normalized   spaces", 'text'));
    $this->assertNull($this->mapper->convertValue('', 'text'));
    $this->assertNull($this->mapper->convertValue(NULL, 'text'));
  }

  /**
   * Tests convertValue() for text type with TextValue object.
   */
  public function testConvertValueTextWithTextValueObject(): void {
    $textValue = new class {

      /**
       * Returns the text content.
       */
      public function getText(): string {
        return 'text from object';
      }

    };

    $this->assertSame('text from object', $this->mapper->convertValue($textValue, 'text'));
  }

  /**
   * Tests convertValue() for text type with non-scalar.
   */
  public function testConvertValueTextWithNonScalar(): void {
    $this->assertNull($this->mapper->convertValue(['array'], 'text'));
  }

  /**
   * Tests convertValue() for string type.
   */
  public function testConvertValueString(): void {
    $this->assertSame('hello', $this->mapper->convertValue('hello', 'string'));
    $this->assertSame('123', $this->mapper->convertValue(123, 'string'));
    $this->assertNull($this->mapper->convertValue('', 'string'));
    $this->assertNull($this->mapper->convertValue(NULL, 'string'));
  }

  /**
   * Tests convertValue() for string type with non-scalar.
   */
  public function testConvertValueStringWithNonScalar(): void {
    $this->assertNull($this->mapper->convertValue(['array'], 'string'));
    $this->assertNull($this->mapper->convertValue(new \stdClass(), 'string'));
  }

  /**
   * Tests convertValue() for string type truncates long values.
   */
  public function testConvertValueStringTruncatesLongValues(): void {
    $long_string = str_repeat('a', 2000);
    $result = $this->mapper->convertValue($long_string, 'string');
    $this->assertSame(1024, strlen((string) $result));
  }

  /**
   * Tests convertValue() for integer type.
   */
  public function testConvertValueInteger(): void {
    $this->assertSame(42, $this->mapper->convertValue(42, 'integer'));
    $this->assertSame(42, $this->mapper->convertValue('42', 'integer'));
    $this->assertSame(42, $this->mapper->convertValue(42.7, 'integer'));
    $this->assertNull($this->mapper->convertValue('not a number', 'integer'));
    $this->assertNull($this->mapper->convertValue('', 'integer'));
  }

  /**
   * Tests convertValue() for boolean type.
   */
  public function testConvertValueBoolean(): void {
    $this->assertSame(1, $this->mapper->convertValue(TRUE, 'boolean'));
    $this->assertSame(0, $this->mapper->convertValue(FALSE, 'boolean'));
    $this->assertSame(1, $this->mapper->convertValue(1, 'boolean'));
    $this->assertSame(0, $this->mapper->convertValue(0, 'boolean'));
    $this->assertSame(1, $this->mapper->convertValue('yes', 'boolean'));
  }

  /**
   * Tests convertValue() for date type.
   */
  public function testConvertValueDate(): void {
    $timestamp = 1700000000;
    $this->assertSame($timestamp, $this->mapper->convertValue($timestamp, 'date'));
    $this->assertSame($timestamp, $this->mapper->convertValue((string) $timestamp, 'date'));

    // Test date string parsing.
    $date_string = '2024-01-15 10:30:00';
    $expected = strtotime($date_string);
    $this->assertSame($expected, $this->mapper->convertValue($date_string, 'date'));
  }

  /**
   * Tests convertValue() for date type with invalid date string.
   */
  public function testConvertValueDateWithInvalidString(): void {
    $this->assertNull($this->mapper->convertValue('not a date', 'date'));
  }

  /**
   * Tests convertValue() for decimal type.
   */
  public function testConvertValueDecimal(): void {
    $this->assertSame(3.14, $this->mapper->convertValue(3.14, 'decimal'));
    $this->assertSame(3.14, $this->mapper->convertValue('3.14', 'decimal'));
    $this->assertSame(42.0, $this->mapper->convertValue(42, 'decimal'));
    $this->assertNull($this->mapper->convertValue('not a number', 'decimal'));
    $this->assertNull($this->mapper->convertValue('', 'decimal'));
  }

  /**
   * Tests convertValue() for uri type.
   */
  public function testConvertValueUri(): void {
    $this->assertSame('https://example.com', $this->mapper->convertValue('https://example.com', 'uri'));
  }

  /**
   * Tests extractFieldValues() with valid values.
   */
  public function testExtractFieldValuesWithValidValues(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getValues')->willReturn(['value1', 'value2', 'value3']);
    $field->method('getType')->willReturn('string');

    $result = $this->mapper->extractFieldValues($field);

    $this->assertCount(3, $result);
    $this->assertSame(['value1', 'value2', 'value3'], $result);
  }

  /**
   * Tests extractFieldValues() filters out null values.
   */
  public function testExtractFieldValuesFiltersNullValues(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getValues')->willReturn(['value1', '', NULL, 'value2']);
    $field->method('getType')->willReturn('string');

    $result = $this->mapper->extractFieldValues($field);

    $this->assertCount(2, $result);
    $this->assertSame(['value1', 'value2'], $result);
  }

  /**
   * Tests extractFieldValues() with empty values.
   */
  public function testExtractFieldValuesWithEmptyValues(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getValues')->willReturn([]);
    $field->method('getType')->willReturn('string');

    $result = $this->mapper->extractFieldValues($field);

    $this->assertSame([], $result);
  }

  /**
   * Tests extractFieldValues() converts values to correct type.
   */
  public function testExtractFieldValuesConvertsTypes(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getValues')->willReturn(['42', '3.14', '100']);
    $field->method('getType')->willReturn('integer');

    $result = $this->mapper->extractFieldValues($field);

    $this->assertCount(3, $result);
    $this->assertSame([42, 3, 100], $result);
  }

}
