<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Index;

use Drupal\search_api\Backend\BackendInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\ServerInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Index\IndexOperations;
use Drupal\search_api_sqlite\Status\StatusReporterInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;

/**
 * Tests the IndexOperations service.
 */
#[CoversClass(IndexOperations::class)]
#[Group('search_api_sqlite')]
final class IndexOperationsTest extends UnitTestCase {

  /**
   * The connection manager mock.
   */
  private ConnectionManagerInterface $connectionManager;

  /**
   * The schema manager mock.
   */
  private SchemaManagerInterface $schemaManager;

  /**
   * The status reporter mock.
   */
  private StatusReporterInterface $statusReporter;

  /**
   * The logger mock.
   */
  private LoggerInterface $logger;

  /**
   * The index operations service under test.
   */
  private IndexOperations $indexOperations;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->statusReporter = $this->createMock(StatusReporterInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->indexOperations = new IndexOperations(
      $this->connectionManager,
      $this->schemaManager,
      $this->statusReporter,
      $this->logger,
    );
  }

  /**
   * Tests getStatistics delegates to StatusReporter.
   */
  public function testGetStatisticsDelegatesToStatusReporter(): void {
    $index_id = 'test_index';
    $expected_stats = [
      'file_path' => '/tmp/test.sqlite',
      'file_size' => 1024,
      'file_size_formatted' => '1.0 KB',
      'indexed_items' => 100,
      'wal_file_exists' => FALSE,
      'wal_file_size' => 0,
      'wal_file_size_formatted' => '0 B',
      'tables' => [],
    ];

    $this->statusReporter->expects($this->once())
      ->method('getDetailedStatistics')
      ->with($index_id)
      ->willReturn($expected_stats);

    $stats = $this->indexOperations->getStatistics($index_id);

    $this->assertEquals($expected_stats, $stats);
  }

  /**
   * Tests optimize throws exception when database doesn't exist.
   */
  public function testOptimizeThrowsExceptionWhenDatabaseNotExists(): void {
    $index_id = 'test_index';

    $this->connectionManager->method('databaseExists')
      ->with($index_id)
      ->willReturn(FALSE);

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Index database does not exist: test_index');

    $this->indexOperations->optimize($index_id);
  }

  /**
   * Tests vacuum throws exception when database doesn't exist.
   */
  public function testVacuumThrowsExceptionWhenDatabaseNotExists(): void {
    $index_id = 'test_index';

    $this->connectionManager->method('databaseExists')
      ->with($index_id)
      ->willReturn(FALSE);

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Index database does not exist: test_index');

    $this->indexOperations->vacuum($index_id);
  }

  /**
   * Tests checkIntegrity when database doesn't exist.
   */
  public function testCheckIntegrityWhenDatabaseNotExists(): void {
    $index_id = 'test_index';

    $this->connectionManager->method('databaseExists')
      ->with($index_id)
      ->willReturn(FALSE);

    $result = $this->indexOperations->checkIntegrity($index_id);

    $this->assertFalse($result['valid']);
    $this->assertContains('Index database does not exist.', $result['messages']);
  }

  /**
   * Tests rebuild throws exception when database doesn't exist.
   */
  public function testRebuildThrowsExceptionWhenDatabaseNotExists(): void {
    $index_id = 'test_index';

    $this->connectionManager->method('databaseExists')
      ->with($index_id)
      ->willReturn(FALSE);

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Index database does not exist: test_index');

    $this->indexOperations->rebuild($index_id);
  }

  /**
   * Tests recreate throws exception when index has no server.
   */
  public function testRecreateThrowsExceptionWhenNoServer(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');
    $index->method('getServerInstance')->willReturn(NULL);

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Index is not attached to a server.');

    $this->indexOperations->recreate($index);
  }

  /**
   * Tests recreate deletes database and recreates schema.
   */
  public function testRecreateDeletesAndRecreatesDatabase(): void {
    $index_id = 'test_index';
    $backend_config = ['tokenizer' => 'unicode61'];

    $backend = $this->createMock(BackendInterface::class);
    $backend->method('getConfiguration')->willReturn($backend_config);

    $server = $this->createMock(ServerInterface::class);
    $server->method('getBackend')->willReturn($backend);

    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn($index_id);
    $index->method('getServerInstance')->willReturn($server);

    // Expect database deletion.
    $this->connectionManager->expects($this->once())
      ->method('deleteDatabase')
      ->with($index_id);

    // Expect schema recreation.
    $this->schemaManager->expects($this->once())
      ->method('createIndexTables')
      ->with($index, $backend_config);

    // Expect reindex to be called.
    $index->expects($this->once())->method('reindex');

    // Expect info log.
    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        $this->stringContains('Recreated SQLite database'),
        $this->arrayHasKey('@index')
      );

    $this->indexOperations->recreate($index);
  }

  /**
   * Tests optimize executes FTS5 optimize command.
   */
  public function testOptimizeExecutesFts5Command(): void {
    $index_id = 'test_index';
    $fts_table = 'test_index_fts';

    $this->connectionManager->method('databaseExists')
      ->with($index_id)
      ->willReturn(TRUE);

    $this->schemaManager->method('getFtsTableName')
      ->with($index_id)
      ->willReturn($fts_table);

    /** @var \PDO&\PHPUnit\Framework\MockObject\MockObject $pdo */
    $pdo = $this->createMock(\PDO::class);
    $pdo->expects($this->once())
      ->method('exec')
      ->with(sprintf("INSERT INTO %s(%s) VALUES('optimize')", $fts_table, $fts_table));

    $this->connectionManager->method('getPdo')
      ->with($index_id)
      ->willReturn($pdo);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        $this->stringContains('Optimized FTS5 index'),
        $this->arrayHasKey('@index')
      );

    $this->indexOperations->optimize($index_id);
  }

  /**
   * Tests vacuum executes VACUUM command.
   */
  public function testVacuumExecutesVacuumCommand(): void {
    $index_id = 'test_index';

    $this->connectionManager->method('databaseExists')
      ->with($index_id)
      ->willReturn(TRUE);

    $pdo = $this->createMock(\PDO::class);
    $pdo->expects($this->once())
      ->method('exec')
      ->with('VACUUM');

    $this->connectionManager->method('getPdo')
      ->with($index_id)
      ->willReturn($pdo);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        $this->stringContains('Vacuumed SQLite database'),
        $this->arrayHasKey('@index')
      );

    $this->indexOperations->vacuum($index_id);
  }

  /**
   * Tests rebuild executes FTS5 rebuild command.
   */
  public function testRebuildExecutesFts5RebuildCommand(): void {
    $index_id = 'test_index';
    $fts_table = 'test_index_fts';

    $this->connectionManager->method('databaseExists')
      ->with($index_id)
      ->willReturn(TRUE);

    $this->schemaManager->method('getFtsTableName')
      ->with($index_id)
      ->willReturn($fts_table);

    $pdo = $this->createMock(\PDO::class);
    $pdo->expects($this->once())
      ->method('exec')
      ->with(sprintf("INSERT INTO %s(%s) VALUES('rebuild')", $fts_table, $fts_table));

    $this->connectionManager->method('getPdo')
      ->with($index_id)
      ->willReturn($pdo);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        $this->stringContains('Rebuilt FTS5 index'),
        $this->arrayHasKey('@index')
      );

    $this->indexOperations->rebuild($index_id);
  }

}
