<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Utility;

use Drupal\search_api\IndexInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;
use Drupal\search_api_sqlite\Utility\IndexSettings;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the IndexSettings utility class.
 */
#[CoversClass(IndexSettings::class)]
#[Group('search_api_sqlite')]
final class IndexSettingsTest extends UnitTestCase {

  /**
   * Tests that defaults() returns the expected structure.
   */
  public function testDefaultsReturnsExpectedStructure(): void {
    $defaults = IndexSettings::defaults();

    // Check top-level keys exist.
    $this->assertArrayHasKey('tokenizer', $defaults);
    $this->assertArrayHasKey('trigram_case_sensitive', $defaults);
    $this->assertArrayHasKey('min_chars', $defaults);
    $this->assertArrayHasKey('matching', $defaults);
    $this->assertArrayHasKey('highlighting', $defaults);
    $this->assertArrayHasKey('optimization', $defaults);

    // Check highlighting structure.
    $this->assertArrayHasKey('enabled', $defaults['highlighting']);
    $this->assertArrayHasKey('prefix', $defaults['highlighting']);
    $this->assertArrayHasKey('suffix', $defaults['highlighting']);
    $this->assertArrayHasKey('excerpt_length', $defaults['highlighting']);

    // Check optimization structure.
    $this->assertArrayHasKey('auto_optimize', $defaults['optimization']);
    $this->assertArrayHasKey('optimize_threshold', $defaults['optimization']);
  }

  /**
   * Tests that defaults() returns expected default values.
   */
  public function testDefaultsReturnsExpectedValues(): void {
    $defaults = IndexSettings::defaults();

    $this->assertEquals(Tokenizer::Unicode61->value, $defaults['tokenizer']);
    $this->assertFalse($defaults['trigram_case_sensitive']);
    $this->assertEquals(3, $defaults['min_chars']);
    $this->assertEquals(MatchingMode::Words->value, $defaults['matching']);
    $this->assertTrue($defaults['highlighting']['enabled']);
    $this->assertEquals('<strong>', $defaults['highlighting']['prefix']);
    $this->assertEquals('</strong>', $defaults['highlighting']['suffix']);
    $this->assertEquals(256, $defaults['highlighting']['excerpt_length']);
    $this->assertTrue($defaults['optimization']['auto_optimize']);
    $this->assertEquals(1000, $defaults['optimization']['optimize_threshold']);
  }

  /**
   * Tests that mergeDefaults() fills in missing values.
   */
  public function testMergeDefaultsFillsMissingValues(): void {
    $partial_settings = [
      'tokenizer' => Tokenizer::Porter->value,
    ];

    $merged = IndexSettings::mergeDefaults($partial_settings);

    // Custom value preserved.
    $this->assertEquals(Tokenizer::Porter->value, $merged['tokenizer']);
    // Default values filled in.
    $this->assertEquals(3, $merged['min_chars']);
    $this->assertArrayHasKey('highlighting', $merged);
    $this->assertTrue($merged['highlighting']['enabled']);
  }

  /**
   * Tests that mergeDefaults() preserves all provided values.
   */
  public function testMergeDefaultsPreservesProvidedValues(): void {
    $custom_settings = [
      'tokenizer' => Tokenizer::Trigram->value,
      'trigram_case_sensitive' => TRUE,
      'min_chars' => 5,
      'matching' => MatchingMode::Prefix->value,
    ];

    $merged = IndexSettings::mergeDefaults($custom_settings);

    $this->assertEquals(Tokenizer::Trigram->value, $merged['tokenizer']);
    $this->assertTrue($merged['trigram_case_sensitive']);
    $this->assertEquals(5, $merged['min_chars']);
    $this->assertEquals(MatchingMode::Prefix->value, $merged['matching']);
  }

  /**
   * Tests that mergeDefaults() deep merges nested arrays.
   */
  public function testMergeDefaultsDeepMergesNestedArrays(): void {
    $partial_settings = [
      'highlighting' => [
        'prefix' => '<mark>',
        // Suffix not provided - should use default.
      ],
    ];

    $merged = IndexSettings::mergeDefaults($partial_settings);

    $this->assertEquals('<mark>', $merged['highlighting']['prefix']);
    // Default suffix should be preserved.
    $this->assertEquals('</strong>', $merged['highlighting']['suffix']);
    $this->assertTrue($merged['highlighting']['enabled']);
  }

  /**
   * Tests getIndexSettings() with an index that has no third-party settings.
   */
  public function testGetIndexSettingsWithNoSettings(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('getThirdPartySettings')
      ->with(IndexSettings::MODULE_NAME)
      ->willReturn([]);

    $settings = IndexSettings::getIndexSettings($index);

    // Should return defaults.
    $this->assertEquals(Tokenizer::Unicode61->value, $settings['tokenizer']);
    $this->assertEquals(3, $settings['min_chars']);
  }

  /**
   * Tests getIndexSettings() with an index that has partial settings.
   */
  public function testGetIndexSettingsWithPartialSettings(): void {
    $stored_settings = [
      'tokenizer' => Tokenizer::Trigram->value,
      'min_chars' => 2,
    ];

    $index = $this->createMock(IndexInterface::class);
    $index->method('getThirdPartySettings')
      ->with(IndexSettings::MODULE_NAME)
      ->willReturn($stored_settings);

    $settings = IndexSettings::getIndexSettings($index);

    // Stored values preserved.
    $this->assertEquals(Tokenizer::Trigram->value, $settings['tokenizer']);
    $this->assertEquals(2, $settings['min_chars']);
    // Defaults filled in.
    $this->assertArrayHasKey('highlighting', $settings);
    $this->assertTrue($settings['highlighting']['enabled']);
  }

  /**
   * Tests getIndexSetting() retrieves a single setting.
   */
  public function testGetIndexSettingRetrievesSingleValue(): void {
    $stored_settings = [
      'tokenizer' => Tokenizer::Porter->value,
    ];

    $index = $this->createMock(IndexInterface::class);
    $index->method('getThirdPartySettings')
      ->with(IndexSettings::MODULE_NAME)
      ->willReturn($stored_settings);

    $tokenizer = IndexSettings::getIndexSetting($index, 'tokenizer');
    $this->assertEquals(Tokenizer::Porter->value, $tokenizer);

    // Non-stored key returns default.
    $min_chars = IndexSettings::getIndexSetting($index, 'min_chars');
    $this->assertEquals(3, $min_chars);
  }

  /**
   * Tests getIndexSetting() returns provided default for missing key.
   */
  public function testGetIndexSettingReturnsProvidedDefault(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('getThirdPartySettings')
      ->with(IndexSettings::MODULE_NAME)
      ->willReturn([]);

    $value = IndexSettings::getIndexSetting($index, 'nonexistent_key', 'custom_default');
    $this->assertEquals('custom_default', $value);
  }

  /**
   * Tests getNestedSetting() retrieves nested values.
   */
  public function testGetNestedSettingRetrievesNestedValues(): void {
    $stored_settings = [
      'highlighting' => [
        'prefix' => '<em>',
      ],
    ];

    $index = $this->createMock(IndexInterface::class);
    $index->method('getThirdPartySettings')
      ->with(IndexSettings::MODULE_NAME)
      ->willReturn($stored_settings);

    // Stored nested value.
    $prefix = IndexSettings::getNestedSetting($index, 'highlighting.prefix');
    $this->assertEquals('<em>', $prefix);

    // Default nested value (suffix not in stored settings).
    $suffix = IndexSettings::getNestedSetting($index, 'highlighting.suffix');
    $this->assertEquals('</strong>', $suffix);
  }

  /**
   * Tests getNestedSetting() returns default for invalid path.
   */
  public function testGetNestedSettingReturnsDefaultForInvalidPath(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('getThirdPartySettings')
      ->with(IndexSettings::MODULE_NAME)
      ->willReturn([]);

    $value = IndexSettings::getNestedSetting($index, 'nonexistent.path.here', 'fallback');
    $this->assertEquals('fallback', $value);
  }

  /**
   * Tests that MODULE_NAME constant is correct.
   */
  public function testModuleNameConstant(): void {
    $this->assertEquals('search_api_sqlite', IndexSettings::MODULE_NAME);
  }

}
