<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Index;

use Drupal\Core\Database\Connection;
use Drupal\Core\State\StateInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Utility\ColumnNameHelper;
use Psr\Log\LoggerInterface;

/**
 * Indexes items into SQLite FTS5 and field data tables.
 *
 * Uses batch operations for efficient indexing:
 * - Batch deletes using IN() clauses instead of per-item deletes.
 * - Batch inserts using Drupal's values() for prepared statement reuse.
 * - Single transaction for atomicity and performance.
 */
final readonly class Indexer implements IndexerInterface {

  /**
   * State key prefix for change tracking.
   */
  private const string STATE_KEY_PREFIX = 'search_api_sqlite.changes.';

  /**
   * Maximum items per batch for IN() clauses.
   *
   * SQLite has SQLITE_MAX_VARIABLE_NUMBER default of 999.
   * Keep batches smaller to leave room for other query parameters.
   */
  private const int BATCH_SIZE = 500;

  /**
   * Constructs an Indexer instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Index\FieldTypeMapperInterface $fieldTypeMapper
   *   The field type mapper.
   * @param \Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface $fts5QueryRunner
   *   The FTS5 query runner.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private ConnectionManagerInterface $connectionManager,
    private SchemaManagerInterface $schemaManager,
    private FieldTypeMapperInterface $fieldTypeMapper,
    private Fts5QueryRunnerInterface $fts5QueryRunner,
    private StateInterface $state,
    private LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line
   */
  public function indexItems(
    IndexInterface $index,
    array $items,
    array $backendConfig,
    ?callable $specialFieldsCallback = NULL,
  ): array {
    if ($items === []) {
      return [];
    }

    $indexId = (string) $index->id();
    $connection = $this->connectionManager->getConnection($indexId);

    $ftsTable = $this->schemaManager->getFtsTableName($indexId);
    $fieldDataTable = $this->schemaManager->getFieldDataTableName($indexId);
    $itemsTable = $this->schemaManager->getItemsTableName($indexId);

    // Categorize fields.
    $fields = $index->getFields();
    $fulltextFields = [];
    $filterFields = [];

    foreach ($fields as $fieldId => $field) {
      if ($this->fieldTypeMapper->isFulltextType($field->getType())) {
        $fulltextFields[$fieldId] = $field;
      }
      else {
        $filterFields[$fieldId] = $field;
      }
    }

    // Start transaction for atomicity and performance.
    $transaction = $connection->startTransaction();

    try {
      // Get all item IDs for batch operations.
      $itemIds = array_map(strval(...), array_keys($items));

      // Batch delete existing data for all items.
      $this->deleteItemsBatch($connection, $ftsTable, $fieldDataTable, $itemsTable, $itemIds);

      // Batch index fulltext fields.
      $this->indexFulltextFieldsBatch($connection, $ftsTable, $items, $fulltextFields);

      // Batch index filter fields.
      $this->indexFilterFieldsBatch($connection, $fieldDataTable, $items, $filterFields);

      // Index special fields if callback provided.
      if ($specialFieldsCallback !== NULL) {
        $this->indexSpecialFieldsBatch($connection, $fieldDataTable, $index, $items, $specialFieldsCallback);
      }

      // Batch track items.
      $this->trackItemsBatch($connection, $itemsTable, $items);

      // Commit transaction (happens when $transaction goes out of scope).
      unset($transaction);

      $this->logger->debug('Indexed @count items for index @index', [
        '@count' => count($itemIds),
        '@index' => $indexId,
      ]);

      // Track changes for auto-optimization.
      $this->trackChanges($indexId, count($itemIds), $backendConfig);

      return $itemIds;
    }
    catch (\Exception $exception) {
      // Rollback happens automatically when $transaction is destroyed.
      $this->logger->error('Failed to index items for @index: @message', [
        '@index' => $indexId,
        '@message' => $exception->getMessage(),
      ]);
      throw $exception;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(IndexInterface $index, array $itemIds): void {
    if ($itemIds === []) {
      return;
    }

    $indexId = (string) $index->id();
    $connection = $this->connectionManager->getConnection($indexId);

    $ftsTable = $this->schemaManager->getFtsTableName($indexId);
    $fieldDataTable = $this->schemaManager->getFieldDataTableName($indexId);
    $itemsTable = $this->schemaManager->getItemsTableName($indexId);

    $transaction = $connection->startTransaction();

    try {
      // Convert to strings for consistency.
      $itemIds = array_map(strval(...), $itemIds);

      // Batch delete.
      $this->deleteItemsBatch($connection, $ftsTable, $fieldDataTable, $itemsTable, $itemIds);

      unset($transaction);

      $this->logger->debug('Deleted @count items from index @index', [
        '@count' => count($itemIds),
        '@index' => $indexId,
      ]);
    }
    catch (\Exception $exception) {
      $this->logger->error('Failed to delete items from @index: @message', [
        '@index' => $indexId,
        '@message' => $exception->getMessage(),
      ]);
      throw $exception;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllItems(IndexInterface $index, ?string $datasourceId = NULL): void {
    $indexId = (string) $index->id();
    $connection = $this->connectionManager->getConnection($indexId);

    $ftsTable = $this->schemaManager->getFtsTableName($indexId);
    $fieldDataTable = $this->schemaManager->getFieldDataTableName($indexId);
    $itemsTable = $this->schemaManager->getItemsTableName($indexId);

    $transaction = $connection->startTransaction();

    try {
      if ($datasourceId === NULL) {
        // Delete all items using Drupal DB API.
        $connection->delete($ftsTable)->execute();
        $connection->delete($fieldDataTable)->execute();
        $connection->delete($itemsTable)->execute();
      }
      else {
        // Get item IDs for this datasource.
        $itemIds = $connection->select($itemsTable, 'i')
          ->fields('i', ['item_id'])
          ->condition('datasource', $datasourceId)
          ->execute()
          ?->fetchCol() ?? [];

        if ($itemIds !== []) {
          $this->deleteItemsBatch($connection, $ftsTable, $fieldDataTable, $itemsTable, $itemIds);
        }
      }

      unset($transaction);

      $this->logger->debug('Deleted all items from index @index (datasource: @datasource)', [
        '@index' => $indexId,
        '@datasource' => $datasourceId ?? 'all',
      ]);
    }
    catch (\Exception $exception) {
      $this->logger->error('Failed to delete all items from @index: @message', [
        '@index' => $indexId,
        '@message' => $exception->getMessage(),
      ]);
      throw $exception;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getIndexedItemsCount(IndexInterface $index): int {
    $indexId = (string) $index->id();

    try {
      $connection = $this->connectionManager->getConnection($indexId);
      $itemsTable = $this->schemaManager->getItemsTableName($indexId);

      $result = $connection->select($itemsTable, 'i')
        ->countQuery()
        ->execute();

      return $result !== NULL ? (int) $result->fetchField() : 0;
    }
    catch (\Exception $exception) {
      $this->logger->error('Error getting indexed items count for @index: @message', [
        '@index' => $indexId,
        '@message' => $exception->getMessage(),
      ]);
      return 0;
    }
  }

  /**
   * Deletes data for multiple items in batches.
   *
   * Uses IN() clauses for efficient batch deletion instead of per-item deletes.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $ftsTable
   *   The FTS5 table name.
   * @param string $fieldDataTable
   *   The field data table name.
   * @param string $itemsTable
   *   The items table name.
   * @param array<string> $itemIds
   *   The item IDs to delete.
   */
  private function deleteItemsBatch(
    Connection $connection,
    string $ftsTable,
    string $fieldDataTable,
    string $itemsTable,
    array $itemIds,
  ): void {
    if ($itemIds === []) {
      return;
    }

    /** @var \PDO $pdo */
    $pdo = $connection->getClientConnection();

    // Process in batches to respect SQLite's variable limit.
    foreach (array_chunk($itemIds, self::BATCH_SIZE) as $batch) {
      $placeholders = implode(',', array_fill(0, count($batch), '?'));

      // FTS5 table - use raw PDO.
      $stmt = $pdo->prepare(sprintf('DELETE FROM %s WHERE item_id IN (%s)', $ftsTable, $placeholders));
      $stmt->execute($batch);

      // Field data table - use Drupal DB API.
      $connection->delete($fieldDataTable)
        ->condition('item_id', $batch, 'IN')
        ->execute();

      // Items table - use Drupal DB API.
      $connection->delete($itemsTable)
        ->condition('item_id', $batch, 'IN')
        ->execute();
    }
  }

  /**
   * Indexes fulltext fields for multiple items.
   *
   * Uses prepared statement reuse via Drupal's values() method.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $ftsTable
   *   The FTS5 table name.
   * @param array<string, \Drupal\search_api\Item\ItemInterface> $items
   *   The items to index.
   * @param array $fulltextFields
   *   The fulltext field definitions keyed by field ID.
   *
   * @phpstan-ignore-next-line
   */
  private function indexFulltextFieldsBatch(
    Connection $connection,
    string $ftsTable,
    array $items,
    array $fulltextFields,
  ): void {
    if ($items === []) {
      return;
    }

    // Build column list.
    $columns = ['item_id'];
    foreach (array_keys($fulltextFields) as $fieldId) {
      $columns[] = ColumnNameHelper::sanitize($fieldId);
    }

    // Use raw PDO for FTS5 - Drupal's insert() can have issues with virtual
    // tables.
    /** @var \PDO $pdo */
    $pdo = $connection->getClientConnection();
    $placeholders = implode(',', array_fill(0, count($columns), '?'));
    $sql = sprintf(
      'INSERT INTO %s (%s) VALUES (%s)',
      $ftsTable,
      implode(',', $columns),
      $placeholders
    );
    $stmt = $pdo->prepare($sql);

    foreach ($items as $itemId => $item) {
      $values = [(string) $itemId];

      foreach (array_keys($fulltextFields) as $fieldId) {
        $itemField = $item->getField($fieldId);
        $text = '';

        if ($itemField !== NULL) {
          $fieldValues = $this->fieldTypeMapper->extractFieldValues($itemField);
          $text = implode(' ', $fieldValues);
        }

        $values[] = $text;
      }

      $stmt->execute($values);
    }
  }

  /**
   * Indexes filter fields for multiple items.
   *
   * Collects all field values and inserts using Drupal's values() for
   * prepared statement reuse.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $fieldDataTable
   *   The field data table name.
   * @param array<string, \Drupal\search_api\Item\ItemInterface> $items
   *   The items to index.
   * @param array $filterFields
   *   The filter field definitions keyed by field ID.
   *
   * @phpstan-ignore-next-line
   */
  private function indexFilterFieldsBatch(
    Connection $connection,
    string $fieldDataTable,
    array $items,
    array $filterFields,
  ): void {
    if ($items === [] || $filterFields === []) {
      return;
    }

    // Collect all rows to insert.
    $rows = [];

    foreach ($items as $itemId => $item) {
      $itemId = (string) $itemId;

      foreach ($filterFields as $fieldId => $fieldDefinition) {
        $itemField = $item->getField($fieldId);
        if ($itemField === NULL) {
          continue;
        }

        $values = $this->fieldTypeMapper->extractFieldValues($itemField);
        $storageType = $this->fieldTypeMapper->getStorageType($fieldDefinition->getType());
        $column = $this->fieldTypeMapper->getFieldDataColumn($storageType);

        foreach ($values as $value) {
          $rows[] = [
            'item_id' => $itemId,
            'field_name' => $fieldId,
            'value_string' => $column === 'value_string' ? $value : NULL,
            'value_integer' => $column === 'value_integer' ? $value : NULL,
            'value_decimal' => $column === 'value_decimal' ? $value : NULL,
          ];
        }
      }
    }

    if ($rows === []) {
      return;
    }

    // Insert using Drupal's values() for prepared statement reuse.
    $query = $connection->insert($fieldDataTable)
      ->fields(['item_id', 'field_name', 'value_string', 'value_integer', 'value_decimal']);

    foreach ($rows as $row) {
      $query->values($row);
    }

    $query->execute();
  }

  /**
   * Indexes special fields for multiple items.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $fieldDataTable
   *   The field data table name.
   * @param \Drupal\search_api\IndexInterface $index
   *   The index.
   * @param array<string, \Drupal\search_api\Item\ItemInterface> $items
   *   The items to index.
   * @param callable $callback
   *   The callback to get special fields.
   *
   * @phpstan-ignore-next-line
   */
  private function indexSpecialFieldsBatch(
    Connection $connection,
    string $fieldDataTable,
    IndexInterface $index,
    array $items,
    callable $callback,
  ): void {
    $rows = [];

    foreach ($items as $itemId => $item) {
      $itemId = (string) $itemId;
      $specialFields = $callback($index, $item);

      foreach ($specialFields as $fieldId => $field) {
        $values = $field->getValues();

        foreach ($values as $value) {
          if ($value !== NULL && $value !== '') {
            $rows[] = [
              'item_id' => $itemId,
              'field_name' => $fieldId,
              'value_string' => (string) $value,
              'value_integer' => NULL,
              'value_decimal' => NULL,
            ];
          }
        }
      }
    }

    if ($rows === []) {
      return;
    }

    $query = $connection->insert($fieldDataTable)
      ->fields(['item_id', 'field_name', 'value_string', 'value_integer', 'value_decimal']);

    foreach ($rows as $row) {
      $query->values($row);
    }

    $query->execute();
  }

  /**
   * Tracks multiple indexed items in the items table.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $itemsTable
   *   The items table name.
   * @param array<string, \Drupal\search_api\Item\ItemInterface> $items
   *   The items.
   *
   * @phpstan-ignore-next-line
   */
  private function trackItemsBatch(
    Connection $connection,
    string $itemsTable,
    array $items,
  ): void {
    if ($items === []) {
      return;
    }

    $now = time();
    $query = $connection->insert($itemsTable)
      ->fields(['item_id', 'datasource', 'language', 'indexed_at']);

    foreach ($items as $itemId => $item) {
      $query->values([
        'item_id' => (string) $itemId,
        'datasource' => $item->getDatasourceId(),
        'language' => $item->getLanguage(),
        'indexed_at' => $now,
      ]);
    }

    $query->execute();
  }

  /**
   * Tracks changes for auto-optimization.
   *
   * Increments the change counter and triggers optimization when the
   * configured threshold is exceeded.
   *
   * @param string $indexId
   *   The index ID.
   * @param int $count
   *   The number of changes.
   * @param array<string, mixed> $backendConfig
   *   The backend configuration.
   */
  private function trackChanges(string $indexId, int $count, array $backendConfig): void {
    $key = self::STATE_KEY_PREFIX . $indexId;
    $current = (int) $this->state->get($key, 0);
    $newCount = $current + $count;
    $this->state->set($key, $newCount);

    // Check if auto-optimization is enabled and threshold is reached.
    $autoOptimize = $backendConfig['optimization']['auto_optimize'] ?? FALSE;
    $threshold = $backendConfig['optimization']['optimize_threshold'] ?? 1000;

    if ($autoOptimize && $newCount >= $threshold) {
      $this->runOptimization($indexId);
    }
  }

  /**
   * Runs FTS5 optimization and resets the change counter.
   *
   * @param string $indexId
   *   The index ID.
   */
  private function runOptimization(string $indexId): void {
    try {
      $ftsTable = $this->schemaManager->getFtsTableName($indexId);
      $this->fts5QueryRunner->optimize($indexId, $ftsTable);

      // Reset the change counter.
      $key = self::STATE_KEY_PREFIX . $indexId;
      $this->state->set($key, 0);

      $this->logger->info('Auto-optimized FTS5 index for @index', [
        '@index' => $indexId,
      ]);
    }
    catch (\Exception $exception) {
      $this->logger->warning('Auto-optimization failed for index @index: @error', [
        '@index' => $indexId,
        '@error' => $exception->getMessage(),
      ]);
    }
  }

}
