<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Index;

use Drupal\Core\State\StateInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\search_api_sqlite\Index\Indexer;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the Indexer auto-optimize functionality.
 *
 * @group search_api_sqlite
 */
#[CoversClass(Indexer::class)]
#[Group('search_api_sqlite')]
class IndexerAutoOptimizeTest extends UnitTestCase {

  /**
   * The mocked connection manager.
   */
  private ConnectionManagerInterface&MockObject $connectionManager;

  /**
   * The mocked schema manager.
   */
  private SchemaManagerInterface&MockObject $schemaManager;

  /**
   * The mocked field type mapper.
   */
  private FieldTypeMapperInterface&MockObject $fieldTypeMapper;

  /**
   * The mocked FTS5 query runner.
   */
  private Fts5QueryRunnerInterface&MockObject $fts5QueryRunner;

  /**
   * The mocked state service.
   */
  private StateInterface&MockObject $state;

  /**
   * The mocked logger.
   */
  private LoggerInterface&MockObject $logger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->fieldTypeMapper = $this->createMock(FieldTypeMapperInterface::class);
    $this->fts5QueryRunner = $this->createMock(Fts5QueryRunnerInterface::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);
  }

  /**
   * Creates an Indexer instance for testing.
   *
   * @return \Drupal\search_api_sqlite\Index\Indexer
   *   The indexer instance.
   */
  private function createIndexer(): Indexer {
    return new Indexer(
      $this->connectionManager,
      $this->schemaManager,
      $this->fieldTypeMapper,
      $this->fts5QueryRunner,
      $this->state,
      $this->logger,
    );
  }

  /**
   * Tests that auto-optimize is triggered when threshold is reached.
   *
   * This is a simplified test using reflection to test the private method.
   */
  public function testAutoOptimizeTriggeredAtThreshold(): void {
    $indexId = 'test_index';
    $ftsTable = 'test_index_fts';

    // Current count is 900, we're adding 200 -> 1100, over 1000 threshold.
    $this->state->expects($this->once())
      ->method('get')
      ->with('search_api_sqlite.changes.' . $indexId, 0)
      ->willReturn(900);

    // First set: increment to 1100, second set: reset to 0 after optimization.
    $this->state->expects($this->exactly(2))
      ->method('set')
      ->willReturnCallback(function ($key, $value) use ($indexId): void {
        static $callCount = 0;
        $callCount++;

        if ($callCount === 1) {
          // First call: incrementing counter.
          $this->assertEquals('search_api_sqlite.changes.' . $indexId, $key);
          $this->assertEquals(1100, $value);
        }
        else {
          // Second call: resetting counter after optimization.
          $this->assertEquals('search_api_sqlite.changes.' . $indexId, $key);
          $this->assertEquals(0, $value);
        }
      });

    $this->schemaManager->expects($this->once())
      ->method('getFtsTableName')
      ->with($indexId)
      ->willReturn($ftsTable);

    $this->fts5QueryRunner->expects($this->once())
      ->method('optimize')
      ->with($indexId, $ftsTable);

    $this->logger->expects($this->once())
      ->method('info')
      ->with('Auto-optimized FTS5 index for @index', ['@index' => $indexId]);

    // Use reflection to test the private trackChanges method.
    $indexer = $this->createIndexer();
    $method = new \ReflectionMethod($indexer, 'trackChanges');

    $backendConfig = [
      'optimization' => [
        'auto_optimize' => TRUE,
        'optimize_threshold' => 1000,
      ],
    ];

    $method->invoke($indexer, $indexId, 200, $backendConfig);
  }

  /**
   * Tests that auto-optimize is not triggered when disabled.
   */
  public function testAutoOptimizeNotTriggeredWhenDisabled(): void {
    $indexId = 'test_index';

    // Will be called to increment, but not to reset.
    $this->state->expects($this->once())
      ->method('get')
      ->with('search_api_sqlite.changes.' . $indexId, 0)
      ->willReturn(900);

    $this->state->expects($this->once())
      ->method('set')
      ->with('search_api_sqlite.changes.' . $indexId, 1100);

    // Optimize should NOT be called.
    $this->fts5QueryRunner->expects($this->never())
      ->method('optimize');

    $indexer = $this->createIndexer();
    $method = new \ReflectionMethod($indexer, 'trackChanges');

    $backendConfig = [
      'optimization' => [
        'auto_optimize' => FALSE,
        'optimize_threshold' => 1000,
      ],
    ];

    $method->invoke($indexer, $indexId, 200, $backendConfig);
  }

  /**
   * Tests that auto-optimize is not triggered below threshold.
   */
  public function testAutoOptimizeNotTriggeredBelowThreshold(): void {
    $indexId = 'test_index';

    // Current count is 500, we're adding 100 -> 600, below 1000 threshold.
    $this->state->expects($this->once())
      ->method('get')
      ->with('search_api_sqlite.changes.' . $indexId, 0)
      ->willReturn(500);

    $this->state->expects($this->once())
      ->method('set')
      ->with('search_api_sqlite.changes.' . $indexId, 600);

    // Optimize should NOT be called.
    $this->fts5QueryRunner->expects($this->never())
      ->method('optimize');

    $indexer = $this->createIndexer();
    $method = new \ReflectionMethod($indexer, 'trackChanges');

    $backendConfig = [
      'optimization' => [
        'auto_optimize' => TRUE,
        'optimize_threshold' => 1000,
      ],
    ];

    $method->invoke($indexer, $indexId, 100, $backendConfig);
  }

}
