<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Spellcheck;

use Drupal\search_api_sqlite\Spellcheck\SpellCheckHandler;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the SpellCheckHandler service.
 *
 * @group search_api_sqlite
 */
#[CoversClass(SpellCheckHandler::class)]
#[Group('search_api_sqlite')]
class SpellCheckHandlerTest extends UnitTestCase {

  /**
   * The logger mock.
   */
  private LoggerInterface&MockObject $logger;

  /**
   * The spell check handler.
   */
  private SpellCheckHandler $handler;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->logger = $this->createMock(LoggerInterface::class);
    $this->handler = new SpellCheckHandler($this->logger);
  }

  /**
   * Tests isAvailable returns TRUE when library is installed.
   */
  public function testIsAvailableReturnsTrue(): void {
    // The php-spellchecker library should be installed in test environment.
    $this->assertTrue($this->handler->isAvailable());
  }

  /**
   * Tests getAvailableBackends returns valid backends.
   */
  public function testGetAvailableBackendsReturnsValidBackends(): void {
    $backends = $this->handler->getAvailableBackends();
    $valid_backends = ['aspell', 'hunspell', 'pspell'];

    // Result count should be within expected range.
    $this->assertLessThanOrEqual(count($valid_backends), count($backends));

    // Each backend should be a known backend name.
    foreach ($backends as $backend) {
      $this->assertContains($backend, $valid_backends);
    }
  }

  /**
   * Tests checkQuery with unavailable backend returns empty result.
   */
  public function testCheckQueryWithUnavailableBackendReturnsEmptyResult(): void {
    $this->logger->expects($this->once())
      ->method('warning')
      ->with(
        $this->stringContains('Spell check failed'),
        $this->anything()
      );

    $result = $this->handler->checkQuery('test query', 'nonexistent', 'en');

    $this->assertFalse($result['has_misspellings']);
    $this->assertEmpty($result['collation']);
    $this->assertEmpty($result['suggestions']);
  }

  /**
   * Tests checkQuery returns result with known backend.
   */
  public function testCheckQueryReturnsResultWithKnownBackend(): void {
    $backends = $this->handler->getAvailableBackends();
    if ($backends === []) {
      $this->markTestSkipped('No spell check backends available');
    }

    $backend = $backends[0];
    $result = $this->handler->checkQuery('test', $backend, 'en');

    // Result should indicate no misspellings for "test".
    $this->assertFalse($result['has_misspellings']);
  }

  /**
   * Tests checkQuery with correctly spelled word returns no misspellings.
   */
  public function testCheckQueryWithCorrectSpelling(): void {
    $backends = $this->handler->getAvailableBackends();
    if ($backends === []) {
      $this->markTestSkipped('No spell check backends available');
    }

    $backend = $backends[0];
    $result = $this->handler->checkQuery('hello world', $backend, 'en');

    $this->assertFalse($result['has_misspellings']);
    $this->assertEmpty($result['collation']);
    $this->assertEmpty($result['suggestions']);
  }

  /**
   * Tests checkQuery with misspelled word returns suggestions.
   */
  public function testCheckQueryWithMisspelling(): void {
    $backends = $this->handler->getAvailableBackends();
    if ($backends === []) {
      $this->markTestSkipped('No spell check backends available');
    }

    $backend = $backends[0];
    // "mispell" is intentionally misspelled.
    $result = $this->handler->checkQuery('mispell', $backend, 'en');

    $this->assertTrue($result['has_misspellings']);
    // Collation should contain corrected query.
    $this->assertNotEmpty($result['collation']);
    $this->assertStringContainsString('misspell', $result['collation']);
    // Suggestions should be keyed by misspelled word.
    $this->assertNotEmpty($result['suggestions']);
    $this->assertArrayHasKey('mispell', $result['suggestions']);
    $this->assertNotEmpty($result['suggestions']['mispell']);
  }

}
