<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Autocomplete;

use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_autocomplete\SearchInterface;
use Drupal\search_api_autocomplete\Suggestion\SuggestionFactory;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Search\QueryBuilderInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for handling autocomplete suggestions.
 *
 * Uses FTS5 prefix queries to find matching terms and extracts
 * completions from the indexed content.
 */
final readonly class AutocompleteHandler implements AutocompleteHandlerInterface {

  /**
   * Constructs an AutocompleteHandler instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Search\QueryBuilderInterface $queryBuilder
   *   The query builder.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private ConnectionManagerInterface $connectionManager,
    private SchemaManagerInterface $schemaManager,
    private QueryBuilderInterface $queryBuilder,
    private LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getSuggestions(
    QueryInterface $query,
    SearchInterface $search,
    string $incomplete_key,
    string $user_input,
    array $fulltext_fields,
  ): array {
    $suggestions = [];
    $index = $query->getIndex();
    $index_id = (string) $index->id();

    // Check if tables exist.
    if (!$this->schemaManager->tablesExist($index_id)) {
      return [];
    }

    // Early return for empty input.
    if ($fulltext_fields === [] || $incomplete_key === '') {
      return [];
    }

    try {
      $fts_table = $this->schemaManager->getFtsTableName($index_id);
      $connection = $this->connectionManager->getConnection($index_id);

      // Build a prefix match query for the incomplete key.
      $escaped_key = $this->queryBuilder->escapeTerm($incomplete_key);
      // Remove quotes if present (we need raw term for prefix).
      $escaped_key = trim($escaped_key, '"');
      $prefix_query = '"' . $escaped_key . '"*';

      // Get the first fulltext column to search in.
      $columns = array_values($fulltext_fields);
      $column = $columns[0];

      // Get limit from search options.
      $limit = $search->getOption('limit') ?? 10;

      // Fetch more rows than the limit since we deduplicate by completion word.
      // Multiple items may have the same word, so we need extras.
      $query_limit = $limit * 5;

      // Execute simple FTS5 prefix query.
      $db_query = $connection->select($fts_table, 'fts');
      $db_query->fields('fts', ['item_id', $column]);
      $db_query->addExpression(sprintf('bm25(%s)', $fts_table), 'score');
      $db_query->where($fts_table . ' MATCH :query', [':query' => $prefix_query]);
      $db_query->orderBy('score');
      $db_query->range(0, $query_limit);

      $result = $db_query->execute();
      if ($result === NULL) {
        return [];
      }

      // Check if SuggestionFactory is available.
      if (!class_exists(SuggestionFactory::class)) {
        $this->logger->warning('search_api_autocomplete module not available.');
        return [];
      }

      $factory = new SuggestionFactory($user_input);

      // Build the base suggestion text from user input.
      $base_text = trim(preg_replace('/\s+/', ' ', $user_input) ?? '');
      $words = explode(' ', $base_text);
      array_pop($words);
      $prefix = $words === [] ? '' : implode(' ', $words) . ' ';

      // Collect words already in the query to avoid suggesting them.
      $existing_words = array_map(mb_strtolower(...), $words);
      $existing_words[] = mb_strtolower($incomplete_key);
      $existing_words = array_flip($existing_words);

      // Create suggestions from results - deduplicate by completion word.
      $seen_completions = [];
      foreach ($result as $row) {
        // Extract all completions from the indexed content.
        $content = $row->{$column} ?? '';
        $completions = $this->extractCompletions($content, $incomplete_key);

        foreach ($completions as $completion) {
          // Deduplicate by the completion word (case-insensitive).
          $completion_lower = mb_strtolower($completion);
          if (isset($seen_completions[$completion_lower])) {
            continue;
          }

          // Skip if this word is already in the search query.
          if (isset($existing_words[$completion_lower])) {
            continue;
          }

          $seen_completions[$completion_lower] = TRUE;

          $suggested_text = $prefix . $completion;
          $suggestions[] = $factory->createFromSuggestedKeys($suggested_text);

          // Stop once we have enough unique suggestions.
          if (count($suggestions) >= $limit) {
            break 2;
          }
        }
      }
    }
    catch (\Exception $exception) {
      $this->logger->error('Autocomplete failed for index @index: @error', [
        '@index' => $index_id,
        '@error' => $exception->getMessage(),
      ]);
    }

    return $suggestions;
  }

  /**
   * Extracts all completion suggestions from indexed content.
   *
   * @param string $content
   *   The indexed content to search in.
   * @param string $incomplete_key
   *   The incomplete search key.
   *
   * @return string[]
   *   Array of completion suggestions found in content.
   */
  private function extractCompletions(string $content, string $incomplete_key): array {
    if ($content === '') {
      return [];
    }

    // Find all words starting with the incomplete key.
    $pattern = '/\b(' . preg_quote($incomplete_key, '/') . '\w*)\b/iu';
    if (preg_match_all($pattern, $content, $matches)) {
      return array_unique($matches[1]);
    }

    return [];
  }

}
