<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

/**
 * Executes FTS5-specific queries and operations.
 *
 * Uses Drupal Database API where possible (select, expressions).
 * Raw query() only for: FTS5 special commands (optimize, rebuild).
 */
final readonly class Fts5QueryRunner implements Fts5QueryRunnerInterface {

  /**
   * Constructs a Fts5QueryRunner instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   */
  public function __construct(
    private ConnectionManagerInterface $connectionManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function optimize(string $index_id, string $table_name): void {
    $connection = $this->connectionManager->getConnection($index_id);

    // Run FTS5 optimize command via raw query (special FTS5 syntax).
    $sql = sprintf("INSERT INTO %s(%s) VALUES('optimize')", $table_name, $table_name);
    $connection->query($sql);
  }

  /**
   * {@inheritdoc}
   */
  public function rebuild(string $index_id, string $table_name): void {
    $connection = $this->connectionManager->getConnection($index_id);

    // Run FTS5 rebuild command via raw query (special FTS5 syntax).
    $sql = sprintf("INSERT INTO %s(%s) VALUES('rebuild')", $table_name, $table_name);
    $connection->query($sql);
  }

  /**
   * {@inheritdoc}
   */
  public function getDocumentCount(string $index_id, string $table_name): int {
    $connection = $this->connectionManager->getConnection($index_id);

    $result = $connection->select($table_name)
      ->countQuery()
      ->execute();

    return $result !== NULL ? (int) $result->fetchField() : 0;
  }

  /**
   * {@inheritdoc}
   */
  public function getMatchCount(string $index_id, string $table_name, string $match_query): int {
    $connection = $this->connectionManager->getConnection($index_id);

    $result = $connection->select($table_name)
      ->where($table_name . ' MATCH :query', [':query' => $match_query])
      ->countQuery()
      ->execute();

    return $result !== NULL ? (int) $result->fetchField() : 0;
  }

}
