<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

use Drupal\Core\Database\Database;
use Psr\Log\LoggerInterface;

/**
 * Provides file-based query logging for SQLite indexes.
 *
 * Logs all SQL queries executed against an index's SQLite database to a
 * log file in the same directory. Useful for debugging and performance
 * analysis.
 */
final class QueryLogger implements QueryLoggerInterface {

  /**
   * The logging key used for Drupal's database log.
   */
  private const string LOGGING_KEY = 'search_api_sqlite';

  /**
   * The connection key prefix.
   */
  private const string CONNECTION_PREFIX = 'search_api_sqlite_';

  /**
   * Indexes with query logging enabled.
   *
   * @var array<string, bool>
   */
  private array $enabled = [];

  /**
   * Constructs a QueryLogger instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly ConnectionManagerInterface $connectionManager,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function enable(string $index_id): void {
    if (isset($this->enabled[$index_id])) {
      return;
    }

    $connection_key = $this->getConnectionKey($index_id);

    // Ensure connection exists before starting logging.
    if (!Database::getConnectionInfo($connection_key)) {
      return;
    }

    // Start Drupal's database logging for this connection.
    Database::startLog(self::LOGGING_KEY, $connection_key);
    $this->enabled[$index_id] = TRUE;

    $this->logger->debug('Query logging enabled for index @index', [
      '@index' => $index_id,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function disable(string $index_id): void {
    if (!isset($this->enabled[$index_id])) {
      return;
    }

    // Flush any pending logs before disabling.
    $this->flush($index_id);

    unset($this->enabled[$index_id]);
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(string $index_id): bool {
    return isset($this->enabled[$index_id]);
  }

  /**
   * {@inheritdoc}
   */
  public function flush(?string $index_id = NULL): void {
    $indexes_to_flush = $index_id !== NULL
      ? [$index_id => TRUE]
      : $this->enabled;

    foreach (array_keys($indexes_to_flush) as $idx) {
      if (!isset($this->enabled[$idx])) {
        continue;
      }

      $connection_key = $this->getConnectionKey($idx);
      $queries = Database::getLog(self::LOGGING_KEY, $connection_key);

      if ($queries === []) {
        // Re-start logging for next batch.
        Database::startLog(self::LOGGING_KEY, $connection_key);
        continue;
      }

      // Write queries to log file.
      $log_path = $this->getLogPath($idx);
      $handle = @fopen($log_path, 'a');

      if ($handle === FALSE) {
        $this->logger->warning('Could not open query log file: @path', [
          '@path' => $log_path,
        ]);
        // Re-start logging despite the error.
        Database::startLog(self::LOGGING_KEY, $connection_key);
        continue;
      }

      foreach ($queries as $query) {
        $time_ms = isset($query['time']) ? round($query['time'] * 1000, 2) : 0;
        $sql = $query['query'] ?? 'unknown';

        // Interpolate bound parameters into the query for readability.
        if (!empty($query['args']) && is_array($query['args'])) {
          $sql = $this->interpolateQuery($sql, $query['args']);
        }

        // Use a common log format: ISO 8601 timestamp, duration, query.
        $line = sprintf(
          "[%s] [%7.2fms] %s\n",
          date('c'),
          $time_ms,
          $sql
        );
        fwrite($handle, $line);
      }

      fclose($handle);

      // Re-start logging for next batch of queries.
      Database::startLog(self::LOGGING_KEY, $connection_key);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getLogPath(string $index_id): string {
    $db_path = $this->connectionManager->getDatabasePath($index_id);
    return preg_replace('/\.sqlite$/', '.log', $db_path) ?? $db_path . '.log';
  }

  /**
   * Gets the connection key for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return string
   *   The connection key.
   */
  private function getConnectionKey(string $index_id): string {
    return self::CONNECTION_PREFIX . $index_id;
  }

  /**
   * Interpolates query parameters into a SQL query string.
   *
   * This is for logging purposes only - creates a readable query string
   * by replacing placeholders with their actual values.
   *
   * @param string $query
   *   The SQL query with placeholders.
   * @param array<string|int, mixed> $args
   *   The query arguments.
   *
   * @return string
   *   The interpolated query string.
   */
  private function interpolateQuery(string $query, array $args): string {
    // Sort by key length descending to replace longer keys first.
    // This prevents :id from matching before :id_value.
    uksort($args, fn($a, $b): int => strlen((string) $b) - strlen((string) $a));

    foreach ($args as $key => $value) {
      // Format the value for display.
      $formatted = match (TRUE) {
        $value === NULL => 'NULL',
        is_bool($value) => $value ? '1' : '0',
        is_int($value), is_float($value) => (string) $value,
        is_array($value) => '(' . implode(', ', array_map($this->quoteValue(...), $value)) . ')',
        default => $this->quoteValue($value),
      };

      // Handle both :named and positional ? placeholders.
      $placeholder = is_int($key) ? '?' : $key;
      if (is_int($key)) {
        // For positional placeholders, replace only the first occurrence.
        $pos = strpos($query, '?');
        if ($pos !== FALSE) {
          $query = substr_replace($query, $formatted, $pos, 1);
        }
      }
      else {
        // For named placeholders, ensure we match the full placeholder name.
        $query = preg_replace('/' . preg_quote($placeholder, '/') . '\\b/', $formatted, $query) ?? $query;
      }
    }

    return $query;
  }

  /**
   * Quotes a value for display in logs.
   *
   * @param mixed $value
   *   The value to quote.
   *
   * @return string
   *   The quoted value.
   */
  private function quoteValue(mixed $value): string {
    if ($value === NULL) {
      return 'NULL';
    }

    if (is_int($value) || is_float($value)) {
      return (string) $value;
    }

    // Escape single quotes and wrap in quotes.
    return "'" . str_replace("'", "''", (string) $value) . "'";
  }

}
