<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Hook;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;
use Drupal\search_api_sqlite\Utility\IndexSettings;

/**
 * Hook implementations for index form alterations.
 */
final class IndexFormHooks {

  use StringTranslationTrait;

  /**
   * The SQLite backend plugin ID.
   */
  private const string BACKEND_PLUGIN_ID = 'search_api_sqlite';

  /**
   * Constructs an IndexFormHooks instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Implements hook_form_search_api_index_form_alter().
   *
   * Adds SQLite-specific settings to the index form as third-party settings.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   */
  #[Hook('form_search_api_index_form_alter')]
  public function formSearchApiIndexFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Only apply to index add/edit forms.
    if (!in_array($form_id, ['search_api_index_form', 'search_api_index_edit_form'], TRUE)) {
      return;
    }

    // Get visibility conditions for SQLite servers.
    $visibility = $this->getSqliteServerVisibility();
    if ($visibility === []) {
      // No SQLite servers configured, nothing to add.
      return;
    }

    // Get current settings from the index.
    $settings = [];
    /** @var \Drupal\Core\Entity\EntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\search_api\IndexInterface $index */
    $index = $form_object->getEntity();
    if (!$index->isNew()) {
      $settings = $index->getThirdPartySettings(IndexSettings::MODULE_NAME);
    }

    $settings = IndexSettings::mergeDefaults($settings);

    // Add third-party settings container.
    $form['third_party_settings'][IndexSettings::MODULE_NAME] = [
      '#type' => 'details',
      '#title' => $this->t('SQLite FTS5 index options'),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="server"]' => $visibility,
        ],
      ],
    ];

    $container = &$form['third_party_settings'][IndexSettings::MODULE_NAME];

    // Tokenizer settings.
    $container['tokenizer'] = [
      '#type' => 'select',
      '#title' => $this->t('Tokenizer'),
      '#description' => $this->t('<p>Controls how text is split into searchable tokens. Choose based on your content type:</p><ul><li><strong>Unicode61</strong> – Recommended for most sites. Best for multilingual content with proper handling of accents, diacritics, and international characters (e.g., "café" matches "cafe", "naïve" matches "naive").</li><li><strong>Porter Stemmer</strong> – For English-only content. Applies stemming to reduce words to their root form (e.g., "running", "runs", "ran" all match "run"). Improves recall but may reduce precision.</li><li><strong>ASCII</strong> – Basic tokenizer for simple English content. Faster but less flexible. Only use if you have pure ASCII text with no special characters.</li><li><strong>Trigram</strong> – Enables true substring matching anywhere in text. Essential for product codes, SKUs, part numbers, or when users need to find items by typing partial codes (e.g., searching "ABC" matches "123ABC456" or "XYZABCDEF"). <strong>Note:</strong> Requires more storage and indexing time.</li></ul>'),
      '#options' => Tokenizer::formOptions(),
      '#default_value' => $settings['tokenizer'],
    ];

    $container['trigram_case_sensitive'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Case-sensitive matching'),
      '#description' => $this->t('When enabled, searches become case-sensitive ("ABC" will not match "abc"). Only applies to the Trigram tokenizer. Enable this if your product codes or SKUs use case to distinguish between items (e.g., "ABC123" vs "abc123" are different products).'),
      '#default_value' => $settings['trigram_case_sensitive'],
      '#states' => [
        'visible' => [
          ':input[name="third_party_settings[search_api_sqlite][tokenizer]"]' => ['value' => Tokenizer::Trigram->value],
        ],
      ],
    ];

    $container['min_chars'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum word length'),
      '#description' => $this->t('Minimum number of characters required for search terms. Terms shorter than this will be ignored in queries (not during indexing). For example, with a value of 3, searching for "the red car" will only search for "red" and "car". This helps avoid performance issues with very short, common words. Not applicable when using Trigram tokenizer.'),
      '#default_value' => $settings['min_chars'],
      '#min' => 1,
      '#max' => 10,
      '#states' => [
        'invisible' => [
          ':input[name="third_party_settings[search_api_sqlite][tokenizer]"]' => ['value' => Tokenizer::Trigram->value],
        ],
      ],
    ];

    $container['matching'] = [
      '#type' => 'select',
      '#title' => $this->t('Default matching mode'),
      '#description' => $this->t('<p>Controls how search terms are matched against indexed content:</p><ul><li><strong>Match all words (AND)</strong> – All search terms must appear somewhere in the result. Most precise and recommended for general use. Searching "drupal search" finds items containing both "drupal" AND "search".</li><li><strong>Prefix matching</strong> – Matches word beginnings only. Best for autocomplete functionality where users type partial words (e.g., "drup" matches "drupal", "drupalcon", "drupalize").</li><li><strong>Partial matching</strong> – Enables substring matching <strong>only when using Trigram tokenizer</strong>. Allows finding text anywhere within words (e.g., "pal" matches "drupal"). Falls back to prefix matching with other tokenizers. More flexible but potentially slower with large datasets.</li><li><strong>Phrase matching</strong> – Terms must appear consecutively in the exact order specified. Searching "hello world" only matches documents containing that exact phrase, not "world hello" or "hello my world".</li></ul>'),
      '#options' => MatchingMode::formOptions(),
      '#default_value' => $settings['matching'],
    ];

    // Optimization settings.
    $container['optimization'] = [
      '#type' => 'details',
      '#title' => $this->t('Optimization'),
      '#open' => FALSE,
    ];

    $container['optimization']['auto_optimize'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable automatic optimization'),
      '#description' => $this->t('Automatically optimize the FTS5 index structure after a certain number of changes. This merges index segments and improves query performance. Recommended for high-traffic sites or frequently updated content. The optimization runs during indexing operations and may briefly increase processing time.'),
      '#default_value' => $settings['optimization']['auto_optimize'],
    ];

    $container['optimization']['optimize_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Optimization threshold'),
      '#description' => $this->t('Number of indexed/updated/deleted items before triggering automatic optimization. Lower values (100-500) keep the index more optimized but increase processing overhead. Higher values (1000-5000) reduce optimization frequency but may allow index fragmentation. Default of 1000 works well for most sites. Adjust based on your indexing patterns and performance needs.'),
      '#default_value' => $settings['optimization']['optimize_threshold'],
      '#min' => 100,
      '#max' => 10000,
      '#states' => [
        'visible' => [
          ':input[name="third_party_settings[search_api_sqlite][optimization][auto_optimize]"]' => ['checked' => TRUE],
        ],
      ],
    ];
  }

  /**
   * Gets visibility conditions for servers using the SQLite backend.
   *
   * @return list<array{value: string}>
   *   Array of visibility conditions for #states.
   */
  private function getSqliteServerVisibility(): array {
    $visibility = [];

    try {
      /** @var \Drupal\search_api\ServerInterface[] $servers */
      $servers = $this->entityTypeManager
        ->getStorage('search_api_server')
        ->loadMultiple();

      foreach ($servers as $server) {
        if ($server->getBackendId() === self::BACKEND_PLUGIN_ID) {
          $visibility[] = ['value' => (string) $server->id()];
        }
      }
    }
    catch (\Exception) {
      // Ignore exceptions during server loading.
    }

    return $visibility;
  }

}
