<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Plugin\search_api\processor;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\search_api\Attribute\SearchApiProcessor;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Processor\ProcessorPluginBase;
use Drupal\search_api\Query\QueryInterface;

/**
 * Enables FTS5 native highlighting for SQLite backend.
 *
 * This processor activates native FTS5 snippet() highlighting by passing
 * configuration to the backend via query options. The backend then includes
 * snippet expressions in the search query for efficient highlighting.
 *
 * When enabled, this processor also adds the
 * 'search_api_skip_processor_highlight' tag to prevent Search API's built-in
 * Highlight processor from running, as native FTS5 highlighting is more
 * accurate for this backend.
 */
#[SearchApiProcessor(
  id: 'search_api_sqlite_highlight',
  label: new TranslatableMarkup('SQLite FTS5 Highlighting'),
  description: new TranslatableMarkup('Uses native FTS5 snippet() function for fast, accurate highlighting of search results.'),
  stages: [
    'preprocess_query' => 0,
  ],
)]
final class Fts5Highlight extends ProcessorPluginBase implements PluginFormInterface {

  use PluginFormTrait;

  /**
   * {@inheritdoc}
   */
  public static function supportsIndex(IndexInterface $index): bool {
    $server = $index->getServerInstance();
    if ($server === NULL) {
      return FALSE;
    }

    return $server->getBackendId() === 'search_api_sqlite';
  }

  /**
   * {@inheritdoc}
   *
   * @return array{prefix: string, suffix: string, excerpt_length: int, exclude_fields: array<string>}
   *   The default configuration.
   */
  public function defaultConfiguration(): array {
    return [
      'prefix' => '<mark>',
      'suffix' => '</mark>',
      'excerpt_length' => 64,
      'exclude_fields' => [],
    ];
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The built form.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Highlight prefix'),
      '#description' => $this->t('HTML to insert before highlighted terms.'),
      '#default_value' => $this->configuration['prefix'],
      '#required' => TRUE,
    ];

    $form['suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Highlight suffix'),
      '#description' => $this->t('HTML to insert after highlighted terms.'),
      '#default_value' => $this->configuration['suffix'],
      '#required' => TRUE,
    ];

    $form['excerpt_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Excerpt length'),
      '#description' => $this->t('Approximate number of tokens (words) to include in the excerpt.'),
      '#default_value' => $this->configuration['excerpt_length'],
      '#min' => 10,
      '#max' => 500,
      '#required' => TRUE,
    ];

    // Build field options for exclusion.
    $fulltext_fields = [];
    $fields = $this->index->getFields();
    foreach ($this->index->getFulltextFields() as $field_id) {
      if (isset($fields[$field_id])) {
        $fulltext_fields[$field_id] = $fields[$field_id]->getLabel() . ' (' . $field_id . ')';
      }
    }

    $form['exclude_fields'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Exclude fields from highlighting'),
      '#description' => $this->t('Select fields that should not be included in the highlighted excerpt.'),
      '#options' => $fulltext_fields,
      '#default_value' => $this->configuration['exclude_fields'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // Clean up the exclude_fields to only include checked values.
    $excluded = $form_state->getValue('exclude_fields');
    $excluded = array_keys(array_filter($excluded));

    $this->setConfiguration([
      'prefix' => $form_state->getValue('prefix'),
      'suffix' => $form_state->getValue('suffix'),
      'excerpt_length' => (int) $form_state->getValue('excerpt_length'),
      'exclude_fields' => $excluded,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function preprocessSearchQuery(QueryInterface $query): void {
    // Skip if no search keys (no highlighting needed).
    if (empty($query->getKeys())) {
      return;
    }

    // Pass configuration to backend via query option.
    $query->setOption('search_api_sqlite_highlight', $this->configuration);

    // Add tag to skip Search API's built-in Highlight processor.
    // Our native FTS5 highlighting is more accurate for this backend.
    $query->addTag('search_api_skip_processor_highlight');
  }

}
