<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Query\SelectInterface;

/**
 * Helper for building condition subqueries.
 *
 * Provides shared functionality for QueryBuilder and FacetBuilder
 * to avoid code duplication.
 */
interface ConditionHelperInterface {

  /**
   * Gets the appropriate value column for a field.
   *
   * @param string $field_name
   *   The field name.
   * @param array<string, \Drupal\search_api\Item\FieldInterface> $fields
   *   The index field definitions.
   * @param mixed $value
   *   The value (used as fallback for type detection).
   *
   * @phpstan-param array<string, \Drupal\search_api\Item\FieldInterface> $fields
   *
   * @return string
   *   The column name (value_string, value_integer, or value_decimal).
   */
  public function getValueColumnForField(string $field_name, array $fields, mixed $value): string;

  /**
   * Determines the value column based on value type.
   *
   * @param mixed $value
   *   The value to check.
   *
   * @return string
   *   The column name (value_string, value_integer, or value_decimal).
   */
  public function getValueColumn(mixed $value): string;

  /**
   * Builds a subquery for search_api_language field (OR context).
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $index_id
   *   The index ID.
   * @param mixed $value
   *   The language value(s).
   * @param string $operator
   *   The operator.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface|null
   *   The subquery or NULL if invalid.
   */
  public function buildLanguageSubquery(
    $connection,
    string $index_id,
    mixed $value,
    string $operator,
  ): ?SelectInterface;

  /**
   * Builds a condition for search_api_language field (AND context).
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $index_id
   *   The index ID.
   * @param mixed $value
   *   The language value(s).
   * @param string $operator
   *   The operator.
   *
   * @return array{0: \Drupal\Core\Database\Query\SelectInterface|null, 1: string}
   *   Array of [subquery, outer_operator].
   */
  public function buildLanguageCondition(
    $connection,
    string $index_id,
    mixed $value,
    string $operator,
  ): array;

}
