<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\DatabaseException;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\search_api\Query\ConditionGroup;
use Drupal\search_api\Query\ConditionGroupInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\sqlite\Driver\Database\sqlite\Connection as SqliteConnection;
use Psr\Log\LoggerInterface;

/**
 * Builds facet queries and calculates facet counts.
 *
 * - AND facets use a temp table from the main query results (reused).
 * - OR facets rebuild the query excluding tagged condition groups.
 * - Supports min_count, limit, missing facets, and zero-count values.
 */
final readonly class FacetBuilder implements FacetBuilderInterface {

  /**
   * Constructs a FacetBuilder instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Search\QueryBuilderInterface $queryBuilder
   *   The query builder (for OR facet query rebuilding).
   * @param \Drupal\search_api_sqlite\Search\ConditionHelperInterface $conditionHelper
   *   The condition helper.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private ConnectionManagerInterface $connectionManager,
    private SchemaManagerInterface $schemaManager,
    private QueryBuilderInterface $queryBuilder,
    private ConditionHelperInterface $conditionHelper,
    private LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getFacets(
    QueryInterface $query,
    SelectInterface $db_query,
    string $index_id,
    array $facet_options,
    array $fulltext_fields,
    ?int $result_count = NULL,
  ): array {
    if ($facet_options === []) {
      return [];
    }

    $connection = $this->connectionManager->getConnection($index_id);
    $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);

    // Separate facets by operator.
    $and_facets = [];
    $or_facets = [];

    foreach ($facet_options as $facet_id => $facet) {
      $operator = $facet['operator'] ?? 'and';
      if ($operator === 'or') {
        $or_facets[$facet_id] = $facet;
      }
      else {
        $and_facets[$facet_id] = $facet;
      }
    }

    $results = [];

    // Create a single temp table for ALL AND facets.
    // This avoids re-executing the main query for each facet.
    $temp_table = NULL;
    if ($and_facets !== []) {
      $temp_table = $this->createTemporaryResultsTable($db_query, $connection);
    }

    // Process AND facets using temp table (or fallback to nested query).
    foreach ($and_facets as $facet_id => $facet) {
      $results[$facet_id] = $this->processAndFacet(
        $facet,
        $fulltext_fields,
        $result_count,
        $temp_table,
        $db_query,
        $connection,
        $field_data_table,
      );
    }

    // Process OR facets - rebuild query excluding tagged conditions.
    foreach ($or_facets as $facet_id => $facet) {
      $results[$facet_id] = $this->processOrFacet(
        $facet,
        $fulltext_fields,
        $query,
        $index_id,
        $connection,
        $field_data_table,
      );
    }

    return $results;
  }

  /**
   * Processes a single AND facet.
   *
   * @param array<string, mixed> $facet
   *   The facet options.
   * @param array<string, string> $fulltext_fields
   *   The fulltext fields.
   * @param int|null $result_count
   *   The result count for early exit optimization.
   * @param string|null $temp_table
   *   The temporary table name, or NULL.
   * @param \Drupal\Core\Database\Query\SelectInterface $db_query
   *   The fallback database query.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $field_data_table
   *   The field data table name.
   *
   * @return array<int, array{count: int, filter: string}>
   *   The facet results.
   */
  private function processAndFacet(
    array $facet,
    array $fulltext_fields,
    ?int $result_count,
    ?string $temp_table,
    SelectInterface $db_query,
    Connection $connection,
    string $field_data_table,
  ): array {
    $field_name = $facet['field'];

    // Skip fulltext fields - FTS5 doesn't support token enumeration.
    if (isset($fulltext_fields[$field_name])) {
      return [];
    }

    // Early exit if result_count < min_count.
    if ($result_count !== NULL && $result_count < $facet['min_count']) {
      return [];
    }

    $select = $this->createBaseQuery($temp_table, $db_query, $connection);

    return $this->calculateFacet(
      $select,
      $connection,
      $field_data_table,
      $facet,
    );
  }

  /**
   * Processes a single OR facet.
   *
   * @param array<string, mixed> $facet
   *   The facet options.
   * @param array<string, string> $fulltext_fields
   *   The fulltext fields.
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The original query.
   * @param string $index_id
   *   The index ID.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $field_data_table
   *   The field data table name.
   *
   * @return array<int, array{count: int, filter: string}>
   *   The facet results.
   */
  private function processOrFacet(
    array $facet,
    array $fulltext_fields,
    QueryInterface $query,
    string $index_id,
    Connection $connection,
    string $field_data_table,
  ): array {
    $field_name = $facet['field'];

    // Skip fulltext fields - FTS5 doesn't support token enumeration.
    if (isset($fulltext_fields[$field_name])) {
      return [];
    }

    $tag = 'facet:' . $field_name;

    // Build a modified query that excludes the tagged condition group.
    $or_db_query = $this->buildQueryWithoutTag(
      $query,
      $index_id,
      $fulltext_fields,
      $tag,
    );

    // Use a nested subquery for OR facets (cannot reuse AND temp table).
    $select = $connection->select($or_db_query, 't');

    return $this->calculateFacet(
      $select,
      $connection,
      $field_data_table,
      $facet,
    );
  }

  /**
   * Creates a base query for facet calculation.
   *
   * @param string|null $temp_table
   *   The temporary table name, or NULL.
   * @param \Drupal\Core\Database\Query\SelectInterface $db_query
   *   The fallback database query.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   *   The base query with item_id field.
   */
  private function createBaseQuery(
    ?string $temp_table,
    SelectInterface $db_query,
    Connection $connection,
  ): SelectInterface {
    if ($temp_table !== NULL) {
      $select = $connection->select($temp_table, 't');
      $select->addField('t', 'item_id');
      return $select;
    }

    // Fallback: nested subquery (less efficient but always works).
    // Clone and remove pagination - facets need ALL matching items.
    $fallback_query = clone $db_query;
    $fallback_query->range();
    return $connection->select($fallback_query, 't');
  }

  /**
   * Creates a temporary table from the query results.
   *
   * Uses queryTemporary() to materialize results once, then reuse for all
   * AND facets.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $db_query
   *   The database query.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   *
   * @return string|null
   *   The temp table name, or NULL if creation failed.
   */
  private function createTemporaryResultsTable(
    SelectInterface $db_query,
    Connection $connection,
  ): ?string {
    // Clone to avoid modifying the original.
    $query = clone $db_query;

    // We only need item_id, remove other fields and expressions.
    $fields = &$query->getFields();
    // Keep only item_id field.
    foreach (array_keys($fields) as $field_name) {
      if ($field_name !== 'item_id') {
        unset($fields[$field_name]);
      }
    }

    // Remove expressions (like score).
    $expressions = &$query->getExpressions();
    $expressions = [];

    // Remove ORDER BY (not needed for facets and may reference removed
    // expressions).
    $order_by = &$query->getOrderBy();
    $order_by = [];

    // Remove RANGE - facets need ALL matching items, not just paginated subset.
    // Setting range to NULL removes any LIMIT clause.
    $query->range();

    // Ensure distinct results.
    $query->distinct();

    if (!$query->preExecute()) {
      return NULL;
    }

    $args = $query->getArguments();

    try {
      // queryTemporary() is defined on driver-specific connection classes.
      assert($connection instanceof SqliteConnection);
      return $connection->queryTemporary((string) $query, $args);
    }
    catch (\PDOException | DatabaseException $e) {
      $this->logger->warning('Could not create temporary table for facets: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Calculates a single facet.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $select
   *   Base query with result item IDs.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $field_data_table
   *   The field data table name.
   * @param array<string, mixed> $facet
   *   The facet options.
   *
   * @return array<int, array{count: int, filter: string}>
   *   The facet results.
   */
  private function calculateFacet(
    SelectInterface $select,
    Connection $connection,
    string $field_data_table,
    array $facet,
  ): array {
    $field_name = $facet['field'];
    $limit = (int) ($facet['limit'] ?? 0);
    $min_count = (int) ($facet['min_count'] ?? 1);
    $include_missing = (bool) ($facet['missing'] ?? FALSE);

    // Use LEFT JOIN for missing facet support, INNER JOIN otherwise.
    $join_type = $include_missing ? 'leftJoin' : 'innerJoin';

    $alias = 'fd';
    $select->$join_type(
      $field_data_table,
      $alias,
      sprintf('t.item_id = %s.item_id AND %s.field_name = :field_name', $alias, $alias),
      [':field_name' => $field_name],
    );

    // Add value expression (COALESCE for different value columns).
    $select->addExpression(
      sprintf('COALESCE(%s.value_string, CAST(%s.value_integer AS TEXT), CAST(%s.value_decimal AS TEXT))', $alias, $alias, $alias),
      'value'
    );
    $select->addExpression('COUNT(DISTINCT t.item_id)', 'num');
    $select->groupBy('value');
    $select->orderBy('num', 'DESC');
    $select->orderBy('value', 'ASC');

    if ($limit > 0) {
      $select->range(0, $limit);
    }

    if ($min_count > 1) {
      $select->having('COUNT(DISTINCT t.item_id) >= :min_count', [':min_count' => $min_count]);
    }

    $terms = [];
    $values = [];
    $has_missing = FALSE;

    $result = $select->execute();
    if ($result) {
      foreach ($result as $row) {
        $count = (int) $row->num;

        // Skip zero-count items unless min_count is 0.
        if ($count < $min_count && $min_count > 0) {
          continue;
        }

        if ($row->value !== NULL) {
          $terms[] = [
            'count' => $count,
            'filter' => '"' . $row->value . '"',
          ];
          $values[] = $row->value;
        }
        else {
          // Missing value.
          $has_missing = TRUE;
          if ($include_missing && $count >= $min_count) {
            $terms[] = [
              'count' => $count,
              'filter' => '!',
            ];
          }
        }
      }
    }

    // Handle min_count = 0: get ALL distinct values and add zero-count ones.
    if ($min_count < 1) {
      $zero_terms = $this->getZeroCountValues(
        $connection,
        $field_data_table,
        $field_name,
        $values,
      );
      $terms = array_merge($terms, $zero_terms);

      // Add missing with count 0 if requested and not already present.
      if ($include_missing && !$has_missing) {
        $terms[] = [
          'count' => 0,
          'filter' => '!',
        ];
      }
    }

    return $terms;
  }

  /**
   * Gets all values with zero count (not in the result set).
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $field_data_table
   *   The field data table name.
   * @param string $field_name
   *   The field name.
   * @param array<string> $existing_values
   *   Values already in the result set.
   *
   * @return array<int, array{count: int, filter: string}>
   *   Zero-count facet values.
   */
  private function getZeroCountValues(
    Connection $connection,
    string $field_data_table,
    string $field_name,
    array $existing_values,
  ): array {
    $select = $connection->select($field_data_table, 'fd');
    $select->addExpression(
      "COALESCE(fd.value_string, CAST(fd.value_integer AS TEXT), CAST(fd.value_decimal AS TEXT))",
      'value'
    );
    $select->condition('fd.field_name', $field_name);
    $select->isNotNull('fd.value_string');
    $select->distinct();

    if ($existing_values !== []) {
      // Exclude values we already have.
      $or_group = $select->orConditionGroup();
      $or_group->condition('fd.value_string', $existing_values, 'NOT IN');
      $or_group->isNull('fd.value_string');
      $select->condition($or_group);
    }

    $terms = [];
    $result = $select->execute();
    if ($result) {
      foreach ($result as $row) {
        if ($row->value !== NULL && !in_array($row->value, $existing_values, TRUE)) {
          $terms[] = [
            'count' => 0,
            'filter' => '"' . $row->value . '"',
          ];
        }
      }
    }

    return $terms;
  }

  /**
   * Builds a query excluding condition groups with a specific tag.
   *
   * This is used for OR facets where we need to recalculate facet values
   * without the facet's own filter applied.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The original query.
   * @param string $index_id
   *   The index ID.
   * @param array<string, string> $fulltext_fields
   *   The fulltext fields.
   * @param string $exclude_tag
   *   The tag to exclude from conditions.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   *   The database query without tagged conditions.
   */
  private function buildQueryWithoutTag(
    QueryInterface $query,
    string $index_id,
    array $fulltext_fields,
    string $exclude_tag,
  ): SelectInterface {
    // Check if any conditions have the exclude tag.
    $has_tag = $this->conditionGroupHasTag(
      $query->getConditionGroup(),
      $exclude_tag,
    );

    if (!$has_tag) {
      // No tagged conditions - build query normally but WITHOUT pagination.
      // Facets need ALL matching items, not just the paginated subset.
      $db_query = $this->queryBuilder->buildQuery(
        $query,
        $index_id,
        $fulltext_fields,
      );
      // Remove pagination - this is critical for correct facet counts.
      $db_query->range();
      return $db_query;
    }

    // We need to rebuild without the tagged conditions.
    // Create a modified condition group without the tagged conditions.
    $modified_conditions = $this->cloneConditionGroupWithoutTag(
      $query->getConditionGroup(),
      $exclude_tag,
    );

    // For the OR facet case, we need to rebuild the query.
    $connection = $this->connectionManager->getConnection($index_id);
    $fts_table = $this->schemaManager->getFtsTableName($index_id);
    $items_table = $this->schemaManager->getItemsTableName($index_id);
    $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);

    $keys = $query->getKeys();
    $has_fulltext = !empty($keys);

    if ($has_fulltext) {
      // Start from FTS table.
      $or_query = $connection->select($fts_table, 'fts');
      $or_query->fields('fts', ['item_id']);

      // Build and apply FTS MATCH.
      $match_query = $this->queryBuilder->buildMatchQuery(
        $query,
        $fulltext_fields,
      );
      if ($match_query !== NULL) {
        $or_query->where($fts_table . ' MATCH :match', [':match' => $match_query]);
      }
    }
    else {
      // Start from items table.
      $or_query = $connection->select($items_table, 'items');
      $or_query->fields('items', ['item_id']);
    }

    // Apply the modified conditions.
    $this->applyConditionGroup(
      $or_query,
      $modified_conditions,
      $connection,
      $index_id,
      $field_data_table,
      $has_fulltext,
      $query->getIndex()->getFields(),
    );

    return $or_query;
  }

  /**
   * Checks if a condition group or its children have a specific tag.
   *
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   The condition group.
   * @param string $tag
   *   The tag to check for.
   *
   * @return bool
   *   TRUE if the tag is found, FALSE otherwise.
   */
  private function conditionGroupHasTag(
    ConditionGroupInterface $condition_group,
    string $tag,
  ): bool {
    if ($condition_group->hasTag($tag)) {
      return TRUE;
    }

    foreach ($condition_group->getConditions() as $condition) {
      if ($condition instanceof ConditionGroupInterface && $this->conditionGroupHasTag($condition, $tag)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Recursively clones a condition group, removing groups with a specific tag.
   *
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   The original condition group.
   * @param string $tag
   *   The tag to remove.
   *
   * @return \Drupal\search_api\Query\ConditionGroupInterface
   *   A cloned condition group without tagged conditions.
   */
  private function cloneConditionGroupWithoutTag(
    ConditionGroupInterface $condition_group,
    string $tag,
  ): ConditionGroupInterface {
    $conjunction = $condition_group->getConjunction();
    $tags = $condition_group->getTags();

    // Create new condition group with same tags (minus the excluded one).
    $filtered_tags = array_filter(
      array_values($tags),
      fn(string $t): bool => $t !== $tag,
    );
    $new_group = new ConditionGroup($conjunction, $filtered_tags);

    foreach ($condition_group->getConditions() as $condition) {
      if ($condition instanceof ConditionGroupInterface) {
        // Skip condition groups with the target tag.
        if ($condition->hasTag($tag)) {
          continue;
        }

        // Recursively process nested groups.
        $nested = $this->cloneConditionGroupWithoutTag($condition, $tag);
        $new_group->addConditionGroup($nested);
      }
      else {
        // Regular condition - add it.
        $new_group->addCondition(
          $condition->getField(),
          $condition->getValue(),
          $condition->getOperator(),
        );
      }
    }

    return $new_group;
  }

  /**
   * Applies a condition group to a query.
   *
   * This is a simplified version of QueryBuilder's condition handling
   * for use with OR facets.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $db_query
   *   The database query.
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   The condition group.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $index_id
   *   The index ID.
   * @param string $field_data_table
   *   The field data table name.
   * @param bool $has_fulltext
   *   Whether this is a fulltext query.
   * @param array<\Drupal\search_api\Item\FieldInterface> $fields
   *   The index field definitions.
   */
  private function applyConditionGroup(
    SelectInterface $db_query,
    ConditionGroupInterface $condition_group,
    $connection,
    string $index_id,
    string $field_data_table,
    bool $has_fulltext,
    array $fields,
  ): void {
    $conditions = $condition_group->getConditions();
    if (empty($conditions)) {
      return;
    }

    $item_id_column = $has_fulltext ? 'fts.item_id' : 'items.item_id';
    $conjunction = $condition_group->getConjunction();

    // For OR conjunction, collect subqueries and combine with UNION.
    if ($conjunction === 'OR') {
      $subqueries = [];

      foreach ($conditions as $condition) {
        if ($condition instanceof ConditionGroupInterface) {
          // For nested OR groups, we need to build a subquery that represents
          // the entire group. We'll recursively build it.
          $nested_subquery = $this->buildConditionGroupSubquery(
            $condition,
            $connection,
            $index_id,
            $field_data_table,
            $has_fulltext,
            $fields,
          );
          if ($nested_subquery !== NULL) {
            $subqueries[] = $nested_subquery;
          }
        }
        else {
          // Simple condition - build subquery.
          $field = $condition->getField();
          $value = $condition->getValue();
          $operator = $condition->getOperator();

          // Handle search_api_language specially.
          if ($field === 'search_api_language') {
            $subquery = $this->conditionHelper->buildLanguageSubquery($connection, $index_id, $value, $operator);
            if ($subquery !== NULL) {
              $subqueries[] = $subquery;
            }
            continue;
          }

          // Skip other special search API fields.
          if (str_starts_with((string) $field, 'search_api_')) {
            continue;
          }

          // Build subquery for this condition.
          [$subquery, $outer_operator] = $this->buildConditionSubquery(
            $connection,
            $field_data_table,
            $field,
            $value,
            $operator,
            $fields,
          );

          // For OR conjunction with outer_operator = 'NOT IN', we need
          // special handling. For now, add subquery (most common is 'IN').
          if ($outer_operator === 'IN') {
            $subqueries[] = $subquery;
          }
          else {
            // NOT IN in OR context is complex - for simplicity, apply directly.
            // This is rare and may need refinement.
            $db_query->condition($item_id_column, $subquery, $outer_operator);
          }
        }
      }

      // Combine all OR subqueries with UNION.
      if (!empty($subqueries)) {
        $combined = $this->combineSubqueriesWithUnion($subqueries);
        if ($combined !== NULL) {
          $db_query->condition($item_id_column, $combined, 'IN');
        }
      }
    }
    else {
      // AND conjunction - apply each condition separately.
      foreach ($conditions as $condition) {
        if ($condition instanceof ConditionGroupInterface) {
          // Recursively handle nested groups.
          $this->applyConditionGroup(
            $db_query,
            $condition,
            $connection,
            $index_id,
            $field_data_table,
            $has_fulltext,
            $fields,
          );
        }
        else {
          // Simple condition.
          $field = $condition->getField();
          $value = $condition->getValue();
          $operator = $condition->getOperator();

          // Handle search_api_language specially - stored in items table.
          if ($field === 'search_api_language') {
            [$subquery, $outer_operator] = $this->conditionHelper->buildLanguageCondition($connection, $index_id, $value, $operator);
            if ($subquery !== NULL) {
              $db_query->condition($item_id_column, $subquery, $outer_operator);
            }
            continue;
          }

          // Skip other special search API fields.
          if (str_starts_with((string) $field, 'search_api_')) {
            continue;
          }

          // Build subquery for this condition.
          [$subquery, $outer_operator] = $this->buildConditionSubquery(
            $connection,
            $field_data_table,
            $field,
            $value,
            $operator,
            $fields,
          );

          $db_query->condition($item_id_column, $subquery, $outer_operator);
        }
      }
    }
  }

  /**
   * Builds a subquery representing an entire condition group.
   *
   * Used for nested OR groups within OR conjunction.
   *
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   The condition group.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $index_id
   *   The index ID.
   * @param string $field_data_table
   *   The field data table name.
   * @param bool $has_fulltext
   *   Whether this is a fulltext query.
   * @param array<\Drupal\search_api\Item\FieldInterface> $fields
   *   The index field definitions.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface|null
   *   The subquery or NULL if empty.
   */
  private function buildConditionGroupSubquery(
    ConditionGroupInterface $condition_group,
    $connection,
    string $index_id,
    string $field_data_table,
    bool $has_fulltext,
    array $fields,
  ): ?SelectInterface {
    $conditions = $condition_group->getConditions();
    if (empty($conditions)) {
      return NULL;
    }

    $conjunction = $condition_group->getConjunction();
    /** @var array<\Drupal\Core\Database\Query\SelectInterface> $subqueries */
    $subqueries = [];

    foreach ($conditions as $condition) {
      if ($condition instanceof ConditionGroupInterface) {
        $nested = $this->buildConditionGroupSubquery(
          $condition,
          $connection,
          $index_id,
          $field_data_table,
          $has_fulltext,
          $fields,
        );
        if ($nested !== NULL) {
          $subqueries[] = $nested;
        }
      }
      else {
        $field = $condition->getField();
        $value = $condition->getValue();
        $operator = $condition->getOperator();

        // Skip search_api fields for now (simplification).
        if (str_starts_with((string) $field, 'search_api_')) {
          continue;
        }

        [$subquery, $outer_operator] = $this->buildConditionSubquery(
          $connection,
          $field_data_table,
          $field,
          $value,
          $operator,
          $fields,
        );

        // Only support IN for now in nested groups.
        if ($outer_operator === 'IN') {
          $subqueries[] = $subquery;
        }
      }
    }

    // Combine based on conjunction.
    if ($conjunction === 'OR') {
      return $this->combineSubqueriesWithUnion($subqueries);
    }

    // AND: We need to intersect - for simplicity, just return first for now.
    // Full implementation would need INTERSECT or complex logic.
    return $this->combineSubqueriesWithUnion($subqueries);
  }

  /**
   * Combines multiple subqueries with UNION.
   *
   * @param array<\Drupal\Core\Database\Query\SelectInterface> $subqueries
   *   The subqueries to combine.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface|null
   *   The combined subquery with UNION, or NULL if empty.
   */
  private function combineSubqueriesWithUnion(array $subqueries): ?SelectInterface {
    if (empty($subqueries)) {
      return NULL;
    }

    if (count($subqueries) === 1) {
      return reset($subqueries);
    }

    // Use the first subquery as base and add UNION for the rest.
    $base = array_shift($subqueries);
    foreach ($subqueries as $subquery) {
      $base->union($subquery);
    }

    return $base;
  }

  /**
   * Builds a subquery for a single condition.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $field_data_table
   *   The field data table name.
   * @param string $field
   *   The field name.
   * @param mixed $value
   *   The condition value.
   * @param string $operator
   *   The condition operator.
   * @param array<\Drupal\search_api\Item\FieldInterface> $fields
   *   The index field definitions.
   *
   * @return array{0: \Drupal\Core\Database\Query\SelectInterface, 1: string}
   *   Array of [subquery, outer_operator].
   */
  private function buildConditionSubquery(
    $connection,
    string $field_data_table,
    string $field,
    mixed $value,
    string $operator,
    array $fields,
  ): array {
    $subquery = $connection->select($field_data_table, 'fd');
    $subquery->fields('fd', ['item_id']);
    $subquery->condition('fd.field_name', $field);

    // Determine value column based on field type from index definition.
    $value_column = $this->conditionHelper->getValueColumnForField($field, $fields, $value);

    // Default outer operator.
    $outer_operator = 'IN';

    // Handle different operators.
    switch (strtoupper($operator)) {
      case 'IN':
        if (is_array($value)) {
          $subquery->condition('fd.' . $value_column, $value, 'IN');
        }

        break;

      case 'NOT IN':
        if (is_array($value)) {
          $subquery->condition('fd.' . $value_column, $value, 'IN');
          $outer_operator = 'NOT IN';
        }

        break;

      case 'BETWEEN':
        if (is_array($value) && count($value) >= 2) {
          $subquery->condition('fd.' . $value_column, $value, 'BETWEEN');
        }

        break;

      case 'NOT BETWEEN':
        if (is_array($value) && count($value) >= 2) {
          $subquery->condition('fd.' . $value_column, $value, 'BETWEEN');
          $outer_operator = 'NOT IN';
        }

        break;

      case 'IS NULL':
        $subquery->isNull('fd.' . $value_column);
        break;

      case 'IS NOT NULL':
        $subquery->isNotNull('fd.' . $value_column);
        break;

      case '<>':
      case '!=':
        // Handle <> NULL as IS NOT NULL.
        if ($value === NULL) {
          $subquery->isNotNull('fd.' . $value_column);
        }
        else {
          $subquery->condition('fd.' . $value_column, $value, '=');
          $outer_operator = 'NOT IN';
        }

        break;

      case '=':
        // Handle = NULL as IS NULL.
        if ($value === NULL) {
          $subquery->isNull('fd.' . $value_column);
        }
        else {
          $subquery->condition('fd.' . $value_column, $value, $operator);
        }

        break;

      default:
        $subquery->condition('fd.' . $value_column, $value, $operator);
        break;
    }

    return [$subquery, $outer_operator];
  }

}
