<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Drupal\search_api\Entity\Server;
use Drupal\search_api_sqlite\Plugin\search_api\backend\SqliteFts5;
use Drupal\Tests\search_api\Kernel\BackendTestBase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests index and search capabilities using the SQLite FTS5 search backend.
 *
 * @see \Drupal\search_api_sqlite\Plugin\search_api\backend\SqliteFts5
 */
#[CoversClass(SqliteFts5::class)]
#[Group('search_api_sqlite')]
class BackendTest extends BackendTestBase {

  use SqliteDatabaseCleanupTrait;

  /**
   * {@inheritdoc}
   *
   * @var list<string>
   */
  protected static $modules = [
    'search_api_sqlite',
    'search_api_test_sqlite',
  ];

  /**
   * {@inheritdoc}
   */
  protected $serverId = 'sapi_sqlite_search_server';

  /**
   * {@inheritdoc}
   */
  protected $indexId = 'sapi_sqlite_search_index';

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->installConfig(['search_api_test_sqlite']);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkServerBackend(): void {
    $server = $this->getServer();
    $this->assertInstanceOf(Server::class, $server);

    $backend = $server->getBackend();
    $this->assertInstanceOf(SqliteFts5::class, $backend);

    // Verify the backend configuration has database_path.
    $config = $backend->getConfiguration();
    $this->assertArrayHasKey('database_path', $config);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkBackendSpecificFeatures(): void {
    $this->searchWithRandom();
    $this->checkMatchModes();
    $this->checkTokenizers();
    $this->checkFacetsWithPagination();
  }

  /**
   * Tests that facet counts are not affected by pagination.
   *
   * Facets should count ALL matching documents, not just the paginated subset.
   */
  protected function checkFacetsWithPagination(): void {
    // Get baseline facet counts without pagination.
    $query = $this->buildSearch();
    $facets_config['category'] = [
      'field' => 'category',
      'limit' => 0,
      'min_count' => 1,
      'missing' => FALSE,
      'operator' => 'and',
    ];
    $query->setOption('search_api_facets', $facets_config);
    $query->range(0, 1000);

    $results = $query->execute();
    $total_count = $results->getResultCount();
    $baseline_facets = $results->getExtraData('search_api_facets', [])['category'] ?? [];

    // Verify we have meaningful data: at least 2 items to demonstrate
    // pagination doesn't affect facet counts.
    $this->assertGreaterThanOrEqual(2, $total_count, 'Test requires at least 2 items.');
    $this->assertNotEmpty($baseline_facets, 'Test requires items with category values.');

    // Calculate total items across all facet values.
    $baseline_total = array_sum(array_column($baseline_facets, 'count'));
    $this->assertGreaterThanOrEqual(2, $baseline_total, 'Baseline facets should count multiple items.');

    // Now run the same query with tight pagination (only 1 result).
    $query = $this->buildSearch();
    $query->setOption('search_api_facets', $facets_config);
    $query->range(0, 1);

    $results = $query->execute();

    // Result count should still be the full count (not limited to 1).
    $this->assertEquals($total_count, $results->getResultCount(), 'Result count should reflect total matches, not pagination.');

    // Only 1 result item should be returned due to pagination.
    $this->assertCount(1, $results->getResultItems(), 'Only 1 result item should be returned due to pagination.');

    // Facets should be identical regardless of pagination.
    $paginated_facets = $results->getExtraData('search_api_facets', [])['category'] ?? [];

    // Sort both to compare.
    usort($baseline_facets, [$this, 'facetCompare']);
    usort($paginated_facets, [$this, 'facetCompare']);

    $this->assertEquals(
      $baseline_facets,
      $paginated_facets,
      'Facet counts should not be affected by pagination.'
    );

    // Verify the facet counts exceed the pagination limit of 1.
    $paginated_total = array_sum(array_column($paginated_facets, 'count'));
    $this->assertGreaterThanOrEqual(2, $paginated_total, 'Facet counts should reflect ALL matching items, not just paginated subset.');
  }

  /**
   * Tests searching with random sort.
   */
  protected function searchWithRandom(): void {
    $results = $this->buildSearch(NULL, [], [], FALSE)
      ->sort('search_api_random')
      ->execute();
    $this->assertEquals(5, $results->getResultCount(), 'Random sorting returned all results.');
  }

  /**
   * {@inheritdoc}
   */
  protected function backendSpecificRegressionTests(): void {
  }

  /**
   * Tests different matching modes.
   */
  protected function checkMatchModes(): void {
    // Test partial matching mode.
    $this->setServerMatchMode('partial');
    $this->indexItems($this->indexId);

    // "tes" should match "test" in partial mode.
    $results = $this->buildSearch('tes')->execute();
    $this->assertGreaterThan(0, $results->getResultCount(), 'Partial matching works.');

    // Test prefix matching mode.
    $this->setServerMatchMode('prefix');
    $this->indexItems($this->indexId);

    // "tes" should match "test" in prefix mode.
    $results = $this->buildSearch('tes')->execute();
    $this->assertGreaterThan(0, $results->getResultCount(), 'Prefix matching works.');

    // Test phrase matching mode.
    $this->setServerMatchMode('phrase');
    $this->indexItems($this->indexId);

    // "foo bar" should match entity 1 which has "foo bar baz" in phrase mode.
    $results = $this->buildSearch('foo bar')->execute();
    $this->assertGreaterThan(0, $results->getResultCount(), 'Phrase matching works.');

    // "bar foo" should NOT match in phrase mode (wrong order).
    $results = $this->buildSearch('bar foo')->execute();
    $this->assertEquals(0, $results->getResultCount(), 'Phrase matching respects word order.');

    // Reset to words mode.
    $this->setServerMatchMode('words');
    $this->indexItems($this->indexId);
  }

  /**
   * Sets the index's matching mode via third-party settings.
   *
   * @param string $match_mode
   *   The matching mode: 'words', 'partial', or 'prefix'.
   */
  protected function setServerMatchMode(string $match_mode): void {
    $index = $this->getIndex();
    $index->setThirdPartySetting('search_api_sqlite', 'matching', $match_mode);
    $index->save();

    // Reset caches.
    $this->resetEntityCache();
  }

  /**
   * Tests different tokenizer configurations.
   *
   * Tests that different FTS5 tokenizers behave correctly:
   * - Porter: Stemming (e.g., "tests" matches "test")
   * - Trigram: Substring matching (e.g., "oba" matches "foobar")
   */
  protected function checkTokenizers(): void {
    // Test Porter stemmer tokenizer.
    $this->setIndexTokenizer('porter');
    // Porter needs reindex because it changes the FTS5 schema.
    $this->indexItems($this->indexId);

    // "testing" should match "test" due to stemming (both stem to "test").
    $results = $this->buildSearch('testing')->execute();
    $this->assertGreaterThan(0, $results->getResultCount(), 'Porter stemmer matches "testing" to "test".');

    // Test Trigram tokenizer for substring matching.
    $this->setIndexTokenizer('trigram');
    $this->indexItems($this->indexId);

    // "obar" should match "foobar" (entity 3 body contains "foobar").
    $results = $this->buildSearch('obar')->execute();
    $this->assertGreaterThan(0, $results->getResultCount(), 'Trigram tokenizer enables substring matching.');

    // "bar" as substring should also match "foobar" and other content.
    $results = $this->buildSearch('bar')->execute();
    $this->assertGreaterThan(0, $results->getResultCount(), 'Trigram matches "bar" substring.');

    // Reset to unicode61 (default).
    $this->setIndexTokenizer('unicode61');
    $this->indexItems($this->indexId);
  }

  /**
   * Sets the index's tokenizer via third-party settings.
   *
   * @param string $tokenizer
   *   The tokenizer: 'unicode61', 'porter', 'ascii', or 'trigram'.
   */
  protected function setIndexTokenizer(string $tokenizer): void {
    $index = $this->getIndex();
    $index->setThirdPartySetting('search_api_sqlite', 'tokenizer', $tokenizer);
    $index->save();

    // Reset caches.
    $this->resetEntityCache();
  }

  /**
   * {@inheritdoc}
   *
   * Override: FTS5 does not support token enumeration for fulltext facets.
   * Unlike search_api_db which stores individual tokens in a separate table,
   * FTS5 maintains an internal inverted index that can't be queried for
   * distinct token values. We return an empty array for fulltext field facets.
   */
  protected function regressionTest2469547(): void {
    $query = $this->buildSearch();
    $facets = [];
    $facets['body'] = [
      'field' => 'body',
      'limit' => 0,
      'min_count' => 1,
      'missing' => FALSE,
    ];
    $query->setOption('search_api_facets', $facets);
    $query->addCondition('id', 5, '<>');
    $query->range(0, 0);

    $results = $query->execute();

    // FTS5 doesn't support token enumeration, so we expect empty facets.
    $facets = $results->getExtraData('search_api_facets', [])['body'] ?? [];
    $this->assertEmpty($facets, 'FTS5 returns empty facets for fulltext fields (not supported).');
  }

}
