<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Autocomplete;

use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_autocomplete\SearchInterface;
use Drupal\search_api_sqlite\Autocomplete\AutocompleteHandler;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Search\QueryBuilderInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the AutocompleteHandler service.
 */
#[CoversClass(AutocompleteHandler::class)]
#[Group('search_api_sqlite')]
final class AutocompleteHandlerTest extends UnitTestCase {

  /**
   * The autocomplete handler under test.
   */
  private AutocompleteHandler $handler;

  /**
   * The mocked connection manager.
   */
  private ConnectionManagerInterface&MockObject $connectionManager;

  /**
   * The mocked schema manager.
   */
  private SchemaManagerInterface&MockObject $schemaManager;

  /**
   * The mocked query builder.
   */
  private QueryBuilderInterface&MockObject $queryBuilder;

  /**
   * The mocked logger.
   */
  private LoggerInterface&MockObject $logger;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->queryBuilder = $this->createMock(QueryBuilderInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->handler = new AutocompleteHandler(
      $this->connectionManager,
      $this->schemaManager,
      $this->queryBuilder,
      $this->logger,
    );
  }

  /**
   * Creates a mock query with index.
   *
   * @param string $index_id
   *   The index ID.
   *
   * @return \Drupal\search_api\Query\QueryInterface&\PHPUnit\Framework\MockObject\MockObject
   *   The mocked query.
   */
  private function createMockQuery(string $index_id = 'test_index'): QueryInterface&MockObject {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn($index_id);

    $query = $this->createMock(QueryInterface::class);
    $query->method('getIndex')->willReturn($index);

    return $query;
  }

  /**
   * Creates a mock search configuration.
   *
   * @param int|null $limit
   *   The suggestion limit.
   *
   * @return \Drupal\search_api_autocomplete\SearchInterface&\PHPUnit\Framework\MockObject\MockObject
   *   The mocked search.
   */
  private function createMockSearch(?int $limit = 10): SearchInterface&MockObject {
    $search = $this->createMock(SearchInterface::class);
    $search->method('getOption')->with('limit')->willReturn($limit);

    return $search;
  }

  /**
   * Tests getSuggestions returns empty when tables don't exist.
   */
  public function testGetSuggestionsReturnsEmptyWhenTablesDoNotExist(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->expects($this->once())
      ->method('tablesExist')
      ->with('test_index')
      ->willReturn(FALSE);

    $result = $this->handler->getSuggestions(
      $query,
      $search,
      'dru',
      'dru',
      ['title' => 'title']
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getSuggestions returns empty for empty incomplete key.
   */
  public function testGetSuggestionsReturnsEmptyForEmptyIncompleteKey(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);

    $result = $this->handler->getSuggestions(
      $query,
      $search,
      '',
      '',
      ['title' => 'title']
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getSuggestions returns empty for empty fulltext fields.
   */
  public function testGetSuggestionsReturnsEmptyForEmptyFulltextFields(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);

    $result = $this->handler->getSuggestions(
      $query,
      $search,
      'dru',
      'dru',
      []
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getSuggestions logs error on exception.
   */
  public function testGetSuggestionsLogsErrorOnException(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')
      ->willThrowException(new \Exception('Database error'));

    $this->logger->expects($this->once())
      ->method('error');

    $result = $this->handler->getSuggestions(
      $query,
      $search,
      'dru',
      'dru',
      ['title' => 'title']
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests that escapeTerm is called with the incomplete key.
   */
  public function testEscapeTermIsCalledWithIncompleteKey(): void {
    $query = $this->createMockQuery();
    $search = $this->createMockSearch();

    $this->schemaManager->method('tablesExist')->willReturn(TRUE);
    $this->schemaManager->method('getFtsTableName')->willReturn('test_index_fts');

    // Expect escapeTerm to be called, then throw to stop further processing.
    $this->queryBuilder->expects($this->once())
      ->method('escapeTerm')
      ->with('drupal')
      ->willThrowException(new \Exception('Stop after escapeTerm'));

    $this->handler->getSuggestions(
      $query,
      $search,
      'drupal',
      'drupal',
      ['title' => 'title']
    );
  }

}
