<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Index;

use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Index\IndexOperations;
use Drupal\search_api_sqlite\Status\StatusReporterInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the IndexOperations service.
 */
#[CoversClass(IndexOperations::class)]
#[Group('search_api_sqlite')]
class IndexOperationsTest extends UnitTestCase {

  /**
   * The mocked connection manager.
   */
  private ConnectionManagerInterface&MockObject $connectionManager;

  /**
   * The mocked schema manager.
   */
  private SchemaManagerInterface&MockObject $schemaManager;

  /**
   * The mocked status reporter.
   */
  private StatusReporterInterface&MockObject $statusReporter;

  /**
   * The mocked logger.
   */
  private LoggerInterface&MockObject $logger;

  /**
   * The index operations service under test.
   */
  private IndexOperations $indexOperations;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->statusReporter = $this->createMock(StatusReporterInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->indexOperations = new IndexOperations(
      $this->connectionManager,
      $this->schemaManager,
      $this->statusReporter,
      $this->logger,
    );
  }

  /**
   * Tests getStatistics() delegates to status reporter.
   */
  public function testGetStatistics(): void {
    $index_id = 'test_index';
    $expected_stats = [
      'fts_rows' => 100,
      'field_data_rows' => 500,
      'items_rows' => 100,
    ];

    $this->statusReporter->expects($this->once())
      ->method('getDetailedStatistics')
      ->with($index_id)
      ->willReturn($expected_stats);

    $result = $this->indexOperations->getStatistics($index_id);

    $this->assertEquals($expected_stats, $result);
  }

  /**
   * Tests optimize() executes FTS5 optimize command.
   */
  public function testOptimize(): void {
    $index_id = 'test_index';
    $fts_table = 'test_index_fts';

    $pdo = $this->createMock(\PDO::class);

    $this->connectionManager->expects($this->once())
      ->method('databaseExists')
      ->with($index_id)
      ->willReturn(TRUE);

    $this->schemaManager->expects($this->once())
      ->method('getFtsTableName')
      ->with($index_id)
      ->willReturn($fts_table);

    $this->connectionManager->expects($this->once())
      ->method('getPdo')
      ->with($index_id)
      ->willReturn($pdo);

    $pdo->expects($this->once())
      ->method('exec')
      ->with(sprintf("INSERT INTO %s(%s) VALUES('optimize')", $fts_table, $fts_table))
      ->willReturn(1);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        'Optimized FTS5 index for @index',
        ['@index' => $index_id]
      );

    $this->indexOperations->optimize($index_id);
  }

  /**
   * Tests optimize() throws when database does not exist.
   */
  public function testOptimizeThrowsWhenDatabaseNotExists(): void {
    $index_id = 'test_index';

    $this->connectionManager->expects($this->once())
      ->method('databaseExists')
      ->with($index_id)
      ->willReturn(FALSE);

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Index database does not exist');

    $this->indexOperations->optimize($index_id);
  }

  /**
   * Tests vacuum() executes VACUUM command.
   */
  public function testVacuum(): void {
    $index_id = 'test_index';

    $pdo = $this->createMock(\PDO::class);

    $this->connectionManager->expects($this->once())
      ->method('databaseExists')
      ->with($index_id)
      ->willReturn(TRUE);

    $this->connectionManager->expects($this->once())
      ->method('getPdo')
      ->with($index_id)
      ->willReturn($pdo);

    $pdo->expects($this->once())
      ->method('exec')
      ->with('VACUUM')
      ->willReturn(1);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        'Vacuumed SQLite database for @index',
        ['@index' => $index_id]
      );

    $this->indexOperations->vacuum($index_id);
  }

  /**
   * Tests checkIntegrity() with valid index.
   */
  public function testCheckIntegrityValid(): void {
    $index_id = 'test_index';
    $fts_table = 'test_index_fts';

    $pdo = $this->createMock(\PDO::class);
    $stmt = $this->createMock(\PDOStatement::class);
    $integrityStmt = $this->createMock(\PDOStatement::class);

    $this->connectionManager->expects($this->once())
      ->method('databaseExists')
      ->with($index_id)
      ->willReturn(TRUE);

    $this->schemaManager->expects($this->once())
      ->method('getFtsTableName')
      ->with($index_id)
      ->willReturn($fts_table);

    $this->connectionManager->expects($this->once())
      ->method('getPdo')
      ->with($index_id)
      ->willReturn($pdo);

    // FTS5 integrity check uses prepare + execute.
    $pdo->expects($this->once())
      ->method('prepare')
      ->with(sprintf("INSERT INTO %s(%s) VALUES('integrity-check')", $fts_table, $fts_table))
      ->willReturn($stmt);

    $stmt->expects($this->once())
      ->method('execute')
      ->willReturn(TRUE);

    // SQLite PRAGMA integrity_check.
    $pdo->expects($this->once())
      ->method('query')
      ->with('PRAGMA integrity_check')
      ->willReturn($integrityStmt);

    $integrityStmt->expects($this->once())
      ->method('fetchColumn')
      ->willReturn('ok');

    $result = $this->indexOperations->checkIntegrity($index_id);

    $this->assertTrue($result['valid']);
    $this->assertNotEmpty($result['messages']);
  }

  /**
   * Tests checkIntegrity() with invalid index returns error.
   */
  public function testCheckIntegrityInvalid(): void {
    $index_id = 'test_index';
    $fts_table = 'test_index_fts';

    $pdo = $this->createMock(\PDO::class);
    $stmt = $this->createMock(\PDOStatement::class);
    $integrityStmt = $this->createMock(\PDOStatement::class);

    $this->connectionManager->expects($this->once())
      ->method('databaseExists')
      ->with($index_id)
      ->willReturn(TRUE);

    $this->schemaManager->expects($this->once())
      ->method('getFtsTableName')
      ->with($index_id)
      ->willReturn($fts_table);

    $this->connectionManager->expects($this->once())
      ->method('getPdo')
      ->with($index_id)
      ->willReturn($pdo);

    // FTS5 integrity check uses prepare + execute.
    $pdo->expects($this->once())
      ->method('prepare')
      ->with(sprintf("INSERT INTO %s(%s) VALUES('integrity-check')", $fts_table, $fts_table))
      ->willReturn($stmt);

    $exception = new \PDOException('FTS integrity check failed');
    $stmt->expects($this->once())
      ->method('execute')
      ->willThrowException($exception);

    // SQLite PRAGMA integrity_check still runs.
    $pdo->expects($this->once())
      ->method('query')
      ->with('PRAGMA integrity_check')
      ->willReturn($integrityStmt);

    $integrityStmt->expects($this->once())
      ->method('fetchColumn')
      ->willReturn('ok');

    $result = $this->indexOperations->checkIntegrity($index_id);

    $this->assertFalse($result['valid']);
    $this->assertNotEmpty($result['messages']);
    $this->assertStringContainsString('FTS integrity check failed', $result['messages'][0]);
  }

  /**
   * Tests rebuild() executes FTS5 rebuild command.
   */
  public function testRebuild(): void {
    $index_id = 'test_index';
    $fts_table = 'test_index_fts';

    $pdo = $this->createMock(\PDO::class);

    $this->connectionManager->expects($this->once())
      ->method('databaseExists')
      ->with($index_id)
      ->willReturn(TRUE);

    $this->schemaManager->expects($this->once())
      ->method('getFtsTableName')
      ->with($index_id)
      ->willReturn($fts_table);

    $this->connectionManager->expects($this->once())
      ->method('getPdo')
      ->with($index_id)
      ->willReturn($pdo);

    $pdo->expects($this->once())
      ->method('exec')
      ->with(sprintf("INSERT INTO %s(%s) VALUES('rebuild')", $fts_table, $fts_table))
      ->willReturn(1);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        'Rebuilt FTS5 index for @index',
        ['@index' => $index_id]
      );

    $this->indexOperations->rebuild($index_id);
  }

}
