<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Commands;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api_sqlite\Index\IndexOperationsInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for SQLite FTS5 index operations.
 */
final class SearchApiSqliteCommands extends DrushCommands {

  /**
   * Constructs a SearchApiSqliteCommands instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\search_api_sqlite\Index\IndexOperationsInterface $indexOperations
   *   The index operations service.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly IndexOperationsInterface $indexOperations,
  ) {
    parent::__construct();
  }

  /**
   * Optimize SQLite FTS5 indexes.
   *
   * Merges b-tree segments for improved query performance.
   * Safe to run anytime, recommended after heavy write operations.
   *
   * @param string|null $index_id
   *   The index ID to optimize, or NULL to optimize all SQLite indexes.
   * @param array<string, mixed> $options
   *   Command options.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @command search-api-sqlite:optimize
   * @aliases sapi-sqlite-opt
   *
   * @arg $index_id The index ID to optimize (optional, use --all for all indexes).
   *
   * @option all Optimize all SQLite indexes.
   *
   * @usage drush search-api-sqlite:optimize node_index
   *   Optimize the node_index.
   * @usage drush search-api-sqlite:optimize --all
   *   Optimize all SQLite indexes.
   */
  #[CLI\Command(name: 'search-api-sqlite:optimize', aliases: ['sapi-sqlite-opt'])]
  #[CLI\Argument(name: 'index_id', description: 'The index ID to optimize (optional, use --all for all indexes)')]
  #[CLI\Option(name: 'all', description: 'Optimize all SQLite indexes')]
  #[CLI\Usage(name: 'drush search-api-sqlite:optimize node_index', description: 'Optimize the node_index')]
  #[CLI\Usage(name: 'drush search-api-sqlite:optimize --all', description: 'Optimize all SQLite indexes')]
  public function optimize(?string $index_id = NULL, array $options = ['all' => FALSE]): void {
    $indexes = $this->loadSqliteIndexes($index_id, (bool) $options['all']);

    foreach ($indexes as $index) {
      try {
        $this->indexOperations->optimize((string) $index->id());
        $this->logger()?->success('Optimized {index}', ['index' => $index->label()]);
      }
      catch (\RuntimeException $e) {
        $this->logger()?->error('Failed to optimize {index}: {error}', [
          'index' => $index->label(),
          'error' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Vacuum SQLite databases.
   *
   * Reclaims disk space and defragments the database file.
   * May take time on large indexes.
   *
   * @param string|null $index_id
   *   The index ID to vacuum, or NULL to vacuum all SQLite indexes.
   * @param array<string, mixed> $options
   *   Command options.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @command search-api-sqlite:vacuum
   * @aliases sapi-sqlite-vac
   *
   * @arg $index_id The index ID to vacuum (optional, use --all for all indexes).
   *
   * @option all Vacuum all SQLite indexes.
   *
   * @usage drush search-api-sqlite:vacuum node_index
   *   Vacuum the node_index database.
   * @usage drush search-api-sqlite:vacuum --all
   *   Vacuum all SQLite index databases.
   */
  #[CLI\Command(name: 'search-api-sqlite:vacuum', aliases: ['sapi-sqlite-vac'])]
  #[CLI\Argument(name: 'index_id', description: 'The index ID to vacuum (optional, use --all for all indexes)')]
  #[CLI\Option(name: 'all', description: 'Vacuum all SQLite indexes')]
  #[CLI\Usage(name: 'drush search-api-sqlite:vacuum node_index', description: 'Vacuum the node_index database')]
  #[CLI\Usage(name: 'drush search-api-sqlite:vacuum --all', description: 'Vacuum all SQLite index databases')]
  public function vacuum(?string $index_id = NULL, array $options = ['all' => FALSE]): void {
    $indexes = $this->loadSqliteIndexes($index_id, (bool) $options['all']);

    foreach ($indexes as $index) {
      try {
        $this->indexOperations->vacuum((string) $index->id());
        $this->logger()?->success('Vacuumed {index}', ['index' => $index->label()]);
      }
      catch (\RuntimeException $e) {
        $this->logger()?->error('Failed to vacuum {index}: {error}', [
          'index' => $index->label(),
          'error' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Check FTS5 index integrity.
   *
   * Validates FTS5 and SQLite database integrity.
   * Use to diagnose potential corruption.
   *
   * @param string|null $index_id
   *   The index ID to check, or NULL to check all SQLite indexes.
   * @param array<string, mixed> $options
   *   Command options.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @command search-api-sqlite:check-integrity
   * @aliases sapi-sqlite-check
   *
   * @arg $index_id The index ID to check (optional, use --all for all indexes).
   *
   * @option all Check all SQLite indexes.
   *
   * @usage drush search-api-sqlite:check-integrity node_index
   *   Check integrity of node_index.
   * @usage drush search-api-sqlite:check-integrity --all
   *   Check integrity of all SQLite indexes.
   */
  #[CLI\Command(name: 'search-api-sqlite:check-integrity', aliases: ['sapi-sqlite-check'])]
  #[CLI\Argument(name: 'index_id', description: 'The index ID to check (optional, use --all for all indexes)')]
  #[CLI\Option(name: 'all', description: 'Check all SQLite indexes')]
  #[CLI\Usage(name: 'drush search-api-sqlite:check-integrity node_index', description: 'Check integrity of node_index')]
  #[CLI\Usage(name: 'drush search-api-sqlite:check-integrity --all', description: 'Check integrity of all SQLite indexes')]
  public function checkIntegrity(?string $index_id = NULL, array $options = ['all' => FALSE]): void {
    $indexes = $this->loadSqliteIndexes($index_id, (bool) $options['all']);

    foreach ($indexes as $index) {
      try {
        $result = $this->indexOperations->checkIntegrity((string) $index->id());

        if ($result['valid']) {
          $this->logger()?->success('Integrity check passed for {index}', ['index' => $index->label()]);
        }
        else {
          $this->logger()?->warning('Integrity check found issues in {index}', ['index' => $index->label()]);
        }

        foreach ($result['messages'] as $message) {
          $this->logger()?->info($message);
        }
      }
      catch (\RuntimeException $e) {
        $this->logger()?->error('Failed to check integrity of {index}: {error}', [
          'index' => $index->label(),
          'error' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Rebuild FTS5 index structures.
   *
   * Reconstructs the FTS5 internal structures from content.
   * Use if integrity check fails. Does not require reindexing.
   *
   * @param string|null $index_id
   *   The index ID to rebuild, or NULL to rebuild all SQLite indexes.
   * @param array<string, mixed> $options
   *   Command options.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @command search-api-sqlite:rebuild
   * @aliases sapi-sqlite-rebuild
   *
   * @arg $index_id The index ID to rebuild (optional, use --all for all indexes).
   *
   * @option all Rebuild all SQLite indexes.
   *
   * @usage drush search-api-sqlite:rebuild node_index
   *   Rebuild node_index FTS5 structures.
   * @usage drush search-api-sqlite:rebuild --all
   *   Rebuild all SQLite index FTS5 structures.
   */
  #[CLI\Command(name: 'search-api-sqlite:rebuild', aliases: ['sapi-sqlite-rebuild'])]
  #[CLI\Argument(name: 'index_id', description: 'The index ID to rebuild (optional, use --all for all indexes)')]
  #[CLI\Option(name: 'all', description: 'Rebuild all SQLite indexes')]
  #[CLI\Usage(name: 'drush search-api-sqlite:rebuild node_index', description: 'Rebuild node_index FTS5 structures')]
  #[CLI\Usage(name: 'drush search-api-sqlite:rebuild --all', description: 'Rebuild all SQLite index FTS5 structures')]
  public function rebuild(?string $index_id = NULL, array $options = ['all' => FALSE]): void {
    $indexes = $this->loadSqliteIndexes($index_id, (bool) $options['all']);

    foreach ($indexes as $index) {
      try {
        $this->indexOperations->rebuild((string) $index->id());
        $this->logger()?->success('Rebuilt {index}', ['index' => $index->label()]);
      }
      catch (\RuntimeException $e) {
        $this->logger()?->error('Failed to rebuild {index}: {error}', [
          'index' => $index->label(),
          'error' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Recreate SQLite index database.
   *
   * Deletes and recreates the SQLite database file. All indexed data
   * will be lost and all items will be queued for reindexing.
   * This is a destructive operation.
   *
   * @param string $index_id
   *   The index ID to recreate.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @command search-api-sqlite:recreate
   * @aliases sapi-sqlite-recreate
   *
   * @arg $index_id The index ID to recreate (required).
   *
   * @usage drush search-api-sqlite:recreate node_index
   *   Recreate the node_index database (requires confirmation).
   * @usage drush search-api-sqlite:recreate node_index -y
   *   Recreate without confirmation prompt.
   */
  #[CLI\Command(name: 'search-api-sqlite:recreate', aliases: ['sapi-sqlite-recreate'])]
  #[CLI\Argument(name: 'index_id', description: 'The index ID to recreate (required)')]
  #[CLI\Usage(name: 'drush search-api-sqlite:recreate node_index', description: 'Recreate the node_index database (requires confirmation)')]
  #[CLI\Usage(name: 'drush search-api-sqlite:recreate node_index -y', description: 'Recreate without confirmation prompt')]
  public function recreate(string $index_id): void {
    $indexes = $this->loadSqliteIndexes($index_id, FALSE);
    $index = reset($indexes);

    if (!$index instanceof IndexInterface) {
      throw new \InvalidArgumentException("Index '$index_id' not found.");
    }

    // Confirm destructive operation.
    // @phpstan-ignore-next-line
    if (!$this->io()->confirm(
      sprintf(
        'This will delete all indexed data for "%s" and queue all items for reindexing. Continue?',
        $index->label()
      ),
      FALSE
    )) {
      $this->logger()?->warning('Operation cancelled.');
      return;
    }

    try {
      $this->indexOperations->recreate($index);
      $this->logger()?->success('Recreated {index}. All items queued for reindexing.', ['index' => $index->label()]);
    }
    catch (\RuntimeException $e) {
      $this->logger()?->error('Failed to recreate {index}: {error}', [
        'index' => $index->label(),
        'error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Show SQLite index statistics.
   *
   * Displays file paths, sizes, item counts, and table statistics.
   *
   * @param string|null $index_id
   *   The index ID to show stats for, or NULL for all SQLite indexes.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   *   The formatted statistics.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @command search-api-sqlite:status
   * @aliases sapi-sqlite-status
   *
   * @arg $index_id The index ID to show stats for (optional, shows all if omitted).
   *
   * @field-labels
   *   id: Index ID
   *   name: Name
   *   file_path: File Path
   *   file_size: File Size
   *   indexed_items: Items
   *   wal_exists: WAL
   *
   * @default-fields id,name,file_size,indexed_items,wal_exists
   *
   * @usage drush search-api-sqlite:status
   *   Show statistics for all SQLite indexes.
   * @usage drush search-api-sqlite:status node_index
   *   Show statistics for node_index.
   */
  #[CLI\Command(name: 'search-api-sqlite:status', aliases: ['sapi-sqlite-status'])]
  #[CLI\Argument(name: 'index_id', description: 'The index ID to show stats for (optional, shows all if omitted)')]
  #[CLI\FieldLabels(labels: [
    'id' => 'Index ID',
    'name' => 'Name',
    'file_path' => 'File Path',
    'file_size' => 'File Size',
    'indexed_items' => 'Items',
    'wal_exists' => 'WAL',
  ])]
  #[CLI\DefaultFields(fields: ['id', 'name', 'file_size', 'indexed_items', 'wal_exists'])]
  #[CLI\Usage(name: 'drush search-api-sqlite:status', description: 'Show statistics for all SQLite indexes')]
  #[CLI\Usage(name: 'drush search-api-sqlite:status node_index', description: 'Show statistics for node_index')]
  public function status(?string $index_id = NULL): RowsOfFields {
    $indexes = $this->loadSqliteIndexes($index_id, $index_id === NULL);
    $rows = [];

    foreach ($indexes as $index) {
      try {
        $stats = $this->indexOperations->getStatistics((string) $index->id());

        $rows[] = [
          'id' => $index->id(),
          'name' => $index->label(),
          'file_path' => $stats['file_path'],
          'file_size' => $stats['file_size_formatted'],
          'indexed_items' => number_format($stats['indexed_items']),
          'wal_exists' => $stats['wal_file_exists'] ? 'Yes (' . $stats['wal_file_size_formatted'] . ')' : 'No',
        ];
      }
      catch (\RuntimeException $e) {
        $this->logger()?->error('Failed to get statistics for {index}: {error}', [
          'index' => $index->label(),
          'error' => $e->getMessage(),
        ]);
      }
    }

    return new RowsOfFields($rows);
  }

  /**
   * Load SQLite indexes.
   *
   * @param string|null $index_id
   *   The specific index ID to load, or NULL.
   * @param bool $all
   *   Whether to load all SQLite indexes.
   *
   * @return array<\Drupal\search_api\IndexInterface>
   *   Array of loaded indexes.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function loadSqliteIndexes(?string $index_id, bool $all): array {
    /** @var \Drupal\search_api\IndexInterface[] $indexes */
    $indexes = [];

    $storage = $this->entityTypeManager->getStorage('search_api_index');

    if ($index_id !== NULL) {
      // Load specific index.
      $index = $storage->load($index_id);

      if (!$index instanceof IndexInterface) {
        throw new \InvalidArgumentException("Index '$index_id' not found.");
      }

      $this->validateSqliteIndex($index);
      $indexes[] = $index;
    }
    elseif ($all) {
      // Load all SQLite indexes.
      /** @var \Drupal\search_api\IndexInterface[] $all_indexes */
      $all_indexes = $storage->loadMultiple();

      foreach ($all_indexes as $index) {
        if ($this->isSqliteIndex($index)) {
          $indexes[] = $index;
        }
      }

      if (empty($indexes)) {
        throw new \InvalidArgumentException('No SQLite indexes found.');
      }
    }
    else {
      throw new \InvalidArgumentException('Either specify an index ID or use --all option.');
    }

    return $indexes;
  }

  /**
   * Validate that an index uses the SQLite backend.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index to validate.
   *
   * @throws \InvalidArgumentException
   *   If the index doesn't use the SQLite backend.
   */
  private function validateSqliteIndex(IndexInterface $index): void {
    if (!$this->isSqliteIndex($index)) {
      throw new \InvalidArgumentException(
        sprintf(
          "Index '%s' does not use the SQLite FTS5 backend.",
          $index->id()
        )
      );
    }
  }

  /**
   * Check if an index uses the SQLite backend.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index to check.
   *
   * @return bool
   *   TRUE if the index uses SQLite backend, FALSE otherwise.
   */
  private function isSqliteIndex(IndexInterface $index): bool {
    if (!$index->hasValidServer()) {
      return FALSE;
    }

    try {
      $server = $index->getServerInstance();
      return $server !== NULL && $server->getBackendId() === 'search_api_sqlite';
    }
    catch (\Exception) {
      return FALSE;
    }
  }

}
