<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Index;

use Drupal\search_api\Item\FieldInterface;

/**
 * Defines the interface for mapping Search API field types to SQLite storage.
 */
interface FieldTypeMapperInterface {

  /**
   * Storage type constants.
   */
  public const STORAGE_FTS5 = 'fts5';

  public const STORAGE_STRING = 'string';

  public const STORAGE_INTEGER = 'integer';

  public const STORAGE_DECIMAL = 'decimal';

  /**
   * Checks if a field type should be stored in FTS5.
   *
   * @param string $type
   *   The Search API field type.
   *
   * @return bool
   *   TRUE if the field should be stored in FTS5, FALSE otherwise.
   */
  public function isFulltextType(string $type): bool;

  /**
   * Gets the SQLite storage type for a Search API field type.
   *
   * @param string $type
   *   The Search API field type.
   *
   * @return string
   *   One of the STORAGE_* constants.
   */
  public function getStorageType(string $type): string;

  /**
   * Converts a field value to the appropriate storage format.
   *
   * @param mixed $value
   *   The raw field value.
   * @param string $type
   *   The Search API field type.
   *
   * @return mixed
   *   The converted value suitable for storage.
   */
  public function convertValue(mixed $value, string $type): mixed;

  /**
   * Gets the field data column name for a storage type.
   *
   * @param string $storage_type
   *   The storage type.
   *
   * @return string
   *   The column name in the field_data table.
   */
  public function getFieldDataColumn(string $storage_type): string;

  /**
   * Extracts values from a Search API field.
   *
   * @param \Drupal\search_api\Item\FieldInterface $field
   *   The Search API field.
   *
   * @return array<int, mixed>
   *   Array of extracted values.
   *
   * @phpstan-ignore-next-line
   */
  public function extractFieldValues(FieldInterface $field): array;

}
