<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Index;

use Drupal\search_api\IndexInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Status\StatusReporterInterface;
use Drupal\search_api_sqlite\Utility\IndexSettings;
use Psr\Log\LoggerInterface;

/**
 * Provides maintenance operations for SQLite FTS5 indexes.
 */
final readonly class IndexOperations implements IndexOperationsInterface {

  /**
   * Constructs an IndexOperations instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Status\StatusReporterInterface $statusReporter
   *   The status reporter.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private ConnectionManagerInterface $connectionManager,
    private SchemaManagerInterface $schemaManager,
    private StatusReporterInterface $statusReporter,
    private LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getStatistics(string $index_id): array {
    return $this->statusReporter->getDetailedStatistics($index_id);
  }

  /**
   * {@inheritdoc}
   */
  public function optimize(string $index_id): void {
    if (!$this->connectionManager->databaseExists($index_id)) {
      throw new \RuntimeException(sprintf('Index database does not exist: %s', $index_id));
    }

    $pdo = $this->connectionManager->getPdo($index_id);
    $fts_table = $this->schemaManager->getFtsTableName($index_id);

    try {
      $pdo->exec(sprintf("INSERT INTO %s(%s) VALUES('optimize')", $fts_table, $fts_table));
      $this->logger->info('Optimized FTS5 index for @index', ['@index' => $index_id]);
    }
    catch (\PDOException $pdoException) {
      $this->logger->error('Failed to optimize index @index: @error', [
        '@index' => $index_id,
        '@error' => $pdoException->getMessage(),
      ]);
      throw new \RuntimeException('Optimization failed: ' . $pdoException->getMessage(), 0, $pdoException);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function vacuum(string $index_id): void {
    if (!$this->connectionManager->databaseExists($index_id)) {
      throw new \RuntimeException(sprintf('Index database does not exist: %s', $index_id));
    }

    $pdo = $this->connectionManager->getPdo($index_id);

    try {
      $pdo->exec('VACUUM');
      $this->logger->info('Vacuumed SQLite database for @index', ['@index' => $index_id]);
    }
    catch (\PDOException $pdoException) {
      $this->logger->error('Failed to vacuum index @index: @error', [
        '@index' => $index_id,
        '@error' => $pdoException->getMessage(),
      ]);
      throw new \RuntimeException('Vacuum failed: ' . $pdoException->getMessage(), 0, $pdoException);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function checkIntegrity(string $index_id): array {
    $result = [
      'valid' => FALSE,
      'messages' => [],
    ];

    if (!$this->connectionManager->databaseExists($index_id)) {
      $result['messages'][] = 'Index database does not exist.';
      return $result;
    }

    $pdo = $this->connectionManager->getPdo($index_id);
    $fts_table = $this->schemaManager->getFtsTableName($index_id);

    try {
      // FTS5 integrity-check returns 'ok' if valid, or error messages.
      $stmt = $pdo->prepare(sprintf(
        "INSERT INTO %s(%s) VALUES('integrity-check')",
        $fts_table,
        $fts_table
      ));
      $stmt->execute();

      // If we get here without exception, the index is valid.
      $result['valid'] = TRUE;
      $result['messages'][] = 'FTS5 index integrity check passed.';

      $this->logger->info('Integrity check passed for @index', ['@index' => $index_id]);
    }
    catch (\PDOException $pdoException) {
      $result['messages'][] = 'FTS5 integrity check failed: ' . $pdoException->getMessage();
      $this->logger->warning('Integrity check failed for @index: @error', [
        '@index' => $index_id,
        '@error' => $pdoException->getMessage(),
      ]);
    }

    // Also run SQLite integrity check.
    try {
      $stmt = $pdo->query('PRAGMA integrity_check');
      if ($stmt !== FALSE) {
        $sqlite_result = $stmt->fetchColumn();
        if ($sqlite_result !== 'ok') {
          $result['valid'] = FALSE;
          $result['messages'][] = 'SQLite integrity check: ' . $sqlite_result;
        }
        else {
          $result['messages'][] = 'SQLite database integrity check passed.';
        }
      }
    }
    catch (\PDOException $pdoException) {
      $result['valid'] = FALSE;
      $result['messages'][] = 'SQLite integrity check failed: ' . $pdoException->getMessage();
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function rebuild(string $index_id): void {
    if (!$this->connectionManager->databaseExists($index_id)) {
      throw new \RuntimeException(sprintf('Index database does not exist: %s', $index_id));
    }

    $pdo = $this->connectionManager->getPdo($index_id);
    $fts_table = $this->schemaManager->getFtsTableName($index_id);

    try {
      $pdo->exec(sprintf("INSERT INTO %s(%s) VALUES('rebuild')", $fts_table, $fts_table));
      $this->logger->info('Rebuilt FTS5 index for @index', ['@index' => $index_id]);
    }
    catch (\PDOException $pdoException) {
      $this->logger->error('Failed to rebuild index @index: @error', [
        '@index' => $index_id,
        '@error' => $pdoException->getMessage(),
      ]);
      throw new \RuntimeException('Rebuild failed: ' . $pdoException->getMessage(), 0, $pdoException);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function recreate(IndexInterface $index): void {
    $index_id = (string) $index->id();

    // Get index configuration for schema recreation.
    $config = IndexSettings::getIndexSettings($index);

    // Get backend configuration for database path.
    $server = $index->getServerInstance();
    if ($server === NULL) {
      throw new \RuntimeException('Index is not attached to a server.');
    }

    $backend = $server->getBackend();
    $backend_config = $backend->getConfiguration();
    $config['database_path'] = $backend_config['database_path'] ?? 'private://search_api_sqlite';

    try {
      // Delete the existing database file.
      $this->connectionManager->deleteDatabase($index_id);

      // Recreate the schema.
      $this->schemaManager->createIndexTables($index, $config);

      // Mark all items for reindexing.
      $index->reindex();

      $this->logger->info('Recreated SQLite database for @index. All items queued for reindexing.', [
        '@index' => $index_id,
      ]);
    }
    catch (\Exception $exception) {
      $this->logger->error('Failed to recreate index @index: @error', [
        '@index' => $index_id,
        '@error' => $exception->getMessage(),
      ]);
      throw new \RuntimeException('Recreation failed: ' . $exception->getMessage(), 0, $exception);
    }
  }

}
