<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\search_api\Utility\Utility;
use Drupal\search_api_autocomplete\Entity\Search;
use Drupal\search_api_autocomplete\Suggestion\SuggestionInterface;
use Drupal\search_api_sqlite\Plugin\search_api\backend\SqliteFts5;
use Drupal\Tests\search_api\Functional\ExampleContentTrait;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests autocomplete functionality of the SQLite FTS5 backend.
 *
 * @requires module search_api_autocomplete
 */
#[CoversClass(SqliteFts5::class)]
#[Group('search_api_sqlite')]
class AutocompleteTest extends KernelTestBase {

  use SqliteDatabaseCleanupTrait;
  use ExampleContentTrait;

  /**
   * {@inheritdoc}
   *
   * @var list<string>
   */
  protected static $modules = [
    'entity_test',
    'field',
    'system',
    'filter',
    'text',
    'user',
    'search_api',
    'search_api_autocomplete',
    'search_api_sqlite',
    'search_api_sqlite_test_autocomplete',
    'search_api_test',
    'search_api_test_sqlite',
    'search_api_test_example_content',
  ];

  /**
   * A search server ID.
   */
  protected string $serverId = 'sapi_sqlite_search_server';

  /**
   * A search index ID.
   */
  protected string $indexId = 'sapi_sqlite_search_index';

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->installSchema('search_api', ['search_api_item']);
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('entity_test_mulrev_changed');
    $this->installEntitySchema('search_api_task');
    $this->installConfig('search_api');

    // Do not use a batch for tracking the initial items after creating an
    // index when running the tests via the GUI. Otherwise, it seems Drupal's
    // Batch API gets confused and the test fails.
    if (!Utility::isRunningInCli()) {
      \Drupal::state()->set('search_api_use_tracking_batch', FALSE);
    }

    $this->installConfig([
      'search_api_test_example_content',
      'search_api_test_sqlite',
      'search_api_sqlite_test_autocomplete',
    ]);

    $this->setUpExampleStructure();
    $this->insertExampleContent();

    $this->indexItems($this->indexId);
  }

  /**
   * Tests whether autocomplete suggestions are correctly created.
   */
  public function testAutocompletion(): void {
    /** @var \Drupal\search_api_autocomplete\SearchInterface|null $autocomplete */
    $autocomplete = Search::load('search_api_sqlite_test_autocomplete');
    if ($autocomplete === NULL) {
      $this->fail('Autocomplete search entity not loaded.');
    }

    $index = $autocomplete->getIndex();
    $server = $index->getServerInstance();
    if ($server === NULL) {
      $this->fail('Server instance not found.');
    }

    $backend = $server->getBackend();
    assert($backend instanceof SqliteFts5);

    // Test basic prefix autocomplete.
    // "fo" should match "foo", "foobar", "foobaz", "foobuz".
    $query = $index->query()->range(0, 10);
    $suggestions = $backend->getAutocompleteSuggestions($query, $autocomplete, 'fo', 'fo');

    $this->assertNotEmpty($suggestions, 'Suggestions returned for "fo".');

    // Check that we get expected suggestions.
    $suggestion_keys = $this->getSuggestionKeys($suggestions);
    $this->assertContains('foo', $suggestion_keys, 'Suggestion "foo" found.');

    // Test multi-word autocomplete.
    // When user has typed "foo " and is typing "fo", suggest completions.
    $query = $index->query()
      ->keys('foo')
      ->range(0, 10);
    $suggestions = $backend->getAutocompleteSuggestions($query, $autocomplete, 'fo', 'foo fo');

    // Should get suggestions that combine "foo" with completions.
    $this->assertNotEmpty($suggestions, 'Multi-word suggestions returned.');
  }

  /**
   * Tests autocomplete with different prefixes.
   */
  public function testAutocompletePrefixes(): void {
    /** @var \Drupal\search_api_autocomplete\SearchInterface|null $autocomplete */
    $autocomplete = Search::load('search_api_sqlite_test_autocomplete');
    if ($autocomplete === NULL) {
      $this->fail('Autocomplete search entity not loaded.');
    }

    $index = $autocomplete->getIndex();
    $server = $index->getServerInstance();
    if ($server === NULL) {
      $this->fail('Server instance not found.');
    }

    $backend = $server->getBackend();
    assert($backend instanceof SqliteFts5);

    // Test "ba" prefix - should match "bar", "baz".
    $query = $index->query()->range(0, 10);
    $suggestions = $backend->getAutocompleteSuggestions($query, $autocomplete, 'ba', 'ba');

    $this->assertNotEmpty($suggestions, 'Suggestions returned for "ba".');
    $suggestion_keys = $this->getSuggestionKeys($suggestions);
    $this->assertContains('bar', $suggestion_keys, 'Suggestion "bar" found for "ba".');

    // Test "tes" prefix - should match "test".
    $query = $index->query()->range(0, 10);
    $suggestions = $backend->getAutocompleteSuggestions($query, $autocomplete, 'tes', 'tes');

    $this->assertNotEmpty($suggestions, 'Suggestions returned for "tes".');
    $suggestion_keys = $this->getSuggestionKeys($suggestions);
    $this->assertContains('test', $suggestion_keys, 'Suggestion "test" found for "tes".');
  }

  /**
   * Extracts suggestion keys from suggestion objects.
   *
   * @param \Drupal\search_api_autocomplete\Suggestion\SuggestionInterface[] $suggestions
   *   The suggestions returned by the backend.
   *
   * @return string[]
   *   Array of suggested keys.
   */
  protected function getSuggestionKeys(array $suggestions): array {
    return array_filter(array_map(
      fn(SuggestionInterface $suggestion) => $suggestion->getSuggestedKeys(),
      $suggestions
    ), fn($key): bool => $key !== NULL);
  }

}
