<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Search;

use Drupal\Core\Database\Query\Select;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Query\ResultSetInterface;
use Drupal\search_api_sqlite\Search\Highlighter;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the Highlighter service.
 */
#[CoversClass(Highlighter::class)]
#[Group('search_api_sqlite')]
class HighlighterTest extends UnitTestCase {

  /**
   * The highlighter under test.
   */
  private Highlighter $highlighter;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->highlighter = new Highlighter();
  }

  /**
   * Tests adding snippet expressions to a query.
   */
  public function testAddSnippetExpressions(): void {
    $db_query = $this->createMock(Select::class);
    $fts_table = 'test_index_fts';
    $fulltext_fields = [
      'title' => 'field_title',
      'body' => 'field_body',
    ];
    $config = [
      'prefix' => '<mark>',
      'suffix' => '</mark>',
      'excerpt_length' => 64,
      'exclude_fields' => [],
    ];

    // Expect addExpression to be called twice (once per field).
    $db_query->expects($this->exactly(2))
      ->method('addExpression')
      ->willReturnCallback(function ($expression, $alias) {
        // Verify the expression contains the table name.
        $this->assertStringContainsString('snippet(test_index_fts', $expression);
        $this->assertStringContainsString('<mark>', $expression);
        $this->assertStringContainsString('</mark>', $expression);
        return $this->createMock(Select::class);
      });

    $this->highlighter->addSnippetExpressions(
      $db_query,
      $fts_table,
      $fulltext_fields,
      $config,
    );
  }

  /**
   * Tests that excluded fields are skipped in snippet expressions.
   */
  public function testAddSnippetExpressionsWithExclusion(): void {
    $db_query = $this->createMock(Select::class);
    $fts_table = 'test_index_fts';
    $fulltext_fields = [
      'title' => 'field_title',
      'body' => 'field_body',
    ];
    $config = [
      'prefix' => '<mark>',
      'suffix' => '</mark>',
      'excerpt_length' => 64,
      'exclude_fields' => ['body'],
    ];

    // Expect addExpression to be called once (body is excluded).
    $db_query->expects($this->once())
      ->method('addExpression')
      ->willReturnCallback(function ($expression, $alias) {
        // Should only get title field (column index 0).
        $this->assertStringContainsString('snippet(test_index_fts, 0', $expression);
        return $this->createMock(Select::class);
      });

    $this->highlighter->addSnippetExpressions(
      $db_query,
      $fts_table,
      $fulltext_fields,
      $config,
    );
  }

  /**
   * Tests processing snippet results.
   */
  public function testProcessSnippetResults(): void {
    $fulltext_fields = [
      'title' => 'field_title',
      'body' => 'field_body',
    ];
    $config = [
      'prefix' => '<mark>',
      'suffix' => '</mark>',
      'excerpt_length' => 64,
      'exclude_fields' => [],
    ];

    // Create mock result item.
    $item = $this->createMock(ItemInterface::class);
    $item->expects($this->once())
      ->method('setExcerpt')
      ->with($this->callback(fn($excerpt): bool =>
          // Should contain the highlighted snippet.
          str_contains((string) $excerpt, '<mark>test</mark>')));

    // Create mock result set.
    $results = $this->createMock(ResultSetInterface::class);
    $results->method('getResultItems')
      ->willReturn(['entity:node/1:en' => $item]);

    // Create mock database row with snippet columns.
    $row = (object) [
      'item_id' => 'entity:node/1:en',
      'score' => -1.5,
      'snippet_title' => 'This is a <mark>test</mark> title',
      'snippet_body' => 'Body text without highlights',
    ];

    $this->highlighter->processSnippetResults(
      $results,
      [$row],
      $fulltext_fields,
      $config,
    );
  }

  /**
   * Tests that snippets without highlights are ignored.
   */
  public function testProcessSnippetResultsIgnoresNonHighlighted(): void {
    $fulltext_fields = [
      'title' => 'field_title',
    ];
    $config = [
      'prefix' => '<mark>',
      'suffix' => '</mark>',
      'excerpt_length' => 64,
      'exclude_fields' => [],
    ];

    // Create mock result item that should NOT receive an excerpt.
    $item = $this->createMock(ItemInterface::class);
    $item->expects($this->never())
      ->method('setExcerpt');

    // Create mock result set.
    $results = $this->createMock(ResultSetInterface::class);
    $results->method('getResultItems')
      ->willReturn(['entity:node/1:en' => $item]);

    // Create mock database row without any highlighted content.
    $row = (object) [
      'item_id' => 'entity:node/1:en',
      'score' => -1.5,
      'snippet_title' => 'No highlights here',
    ];

    $this->highlighter->processSnippetResults(
      $results,
      [$row],
      $fulltext_fields,
      $config,
    );
  }

}
