<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Utility;

use Drupal\search_api_sqlite\Utility\FormatHelper;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the FormatHelper utility class.
 */
#[CoversClass(FormatHelper::class)]
#[Group('search_api_sqlite')]
class FormatHelperTest extends UnitTestCase {

  /**
   * Data provider for byte formatting tests.
   *
   * @return array<string, array{int, string}>
   *   Test cases with bytes input and expected output.
   */
  public static function bytesProvider(): array {
    return [
      'zero bytes' => [0, '0 B'],
      'one byte' => [1, '1 B'],
      'bytes' => [500, '500 B'],
      'one kilobyte' => [1024, '1 KB'],
      'kilobytes' => [1536, '1.5 KB'],
      'one megabyte' => [1048576, '1 MB'],
      'megabytes' => [5242880, '5 MB'],
      'one gigabyte' => [1073741824, '1 GB'],
      'gigabytes' => [2147483648, '2 GB'],
      'large size' => [10737418240, '10 GB'],
    ];
  }

  /**
   * Tests formatBytes() with various sizes.
   */
  #[DataProvider('bytesProvider')]
  public function testFormatBytes(int $bytes, string $expected): void {
    $this->assertEquals($expected, FormatHelper::formatBytes($bytes));
  }

  /**
   * Tests formatBytes() with custom precision.
   */
  public function testFormatBytesCustomPrecision(): void {
    // 1.5 KB with different precisions.
    $this->assertEquals('1.5 KB', FormatHelper::formatBytes(1536, 1));
    $this->assertEquals('1.5 KB', FormatHelper::formatBytes(1536, 3));
    $this->assertEquals('2 KB', FormatHelper::formatBytes(1536, 0));
  }

}
