<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

use Drupal\Core\Database\Connection;

/**
 * Defines the interface for SQLite connection management.
 */
interface ConnectionManagerInterface {

  /**
   * Gets or creates a database connection for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string|null $database_path
   *   Optional custom database path override (stream wrapper or absolute).
   *
   * @return \Drupal\Core\Database\Connection
   *   The database connection.
   *
   * @throws \RuntimeException
   *   If the connection cannot be established.
   */
  public function getConnection(string $index_id, ?string $database_path = NULL): Connection;

  /**
   * Closes and removes a database connection.
   *
   * @param string $index_id
   *   The Search API index ID.
   */
  public function closeConnection(string $index_id): void;

  /**
   * Gets the database file path for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return string
   *   The absolute path to the database file.
   */
  public function getDatabasePath(string $index_id): string;

  /**
   * Checks if a database exists for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return bool
   *   TRUE if the database exists, FALSE otherwise.
   */
  public function databaseExists(string $index_id): bool;

  /**
   * Deletes the database file for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return bool
   *   TRUE if the database was deleted, FALSE otherwise.
   */
  public function deleteDatabase(string $index_id): bool;

  /**
   * Gets the raw PDO connection for an index.
   *
   * This is needed for FTS5-specific operations that require raw SQL
   * execution (like optimize, rebuild, integrity-check commands).
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return \PDO
   *   The PDO connection.
   *
   * @throws \RuntimeException
   *   If the connection cannot be established.
   */
  public function getPdo(string $index_id): \PDO;

  /**
   * Creates a temporary table populated with item IDs.
   *
   * This is useful for efficient JOINs when filtering by a large set of
   * item IDs, avoiding SQLite's limit on IN clause parameters and
   * improving query performance.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param array<int, string> $item_ids
   *   The item IDs to insert into the temp table.
   * @param string $purpose
   *   A short identifier for the temp table purpose (e.g., 'facet', 'filter').
   *
   * @return string
   *   The name of the created temp table.
   */
  public function createTempItemsTable(string $index_id, array $item_ids, string $purpose = 'items'): string;

  /**
   * Drops a temporary table.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The name of the temp table to drop.
   */
  public function dropTempTable(string $index_id, string $table_name): void;

  /**
   * Determines if a temp table should be used based on item count.
   *
   * Below the threshold, IN() clauses are faster (no temp table overhead).
   * At or above the threshold, temp table JOINs are more efficient.
   *
   * @param int $count
   *   The number of items.
   *
   * @return bool
   *   TRUE if a temp table should be used, FALSE for IN clause.
   */
  public function shouldUseTempTable(int $count): bool;

  /**
   * Returns the threshold for switching to temp table approach.
   *
   * @return int
   *   The threshold value.
   */
  public function getTempTableThreshold(): int;

}
