<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

use Drupal\search_api\IndexInterface;
use Drupal\search_api_sqlite\Enum\Tokenizer;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\search_api_sqlite\Utility\ColumnNameHelper;

/**
 * Manages SQLite database schema for Search API indexes.
 */
final readonly class SchemaManager implements SchemaManagerInterface {

  /**
   * Constructs a SchemaManager instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Index\FieldTypeMapperInterface $fieldTypeMapper
   *   The field type mapper.
   */
  public function __construct(
    private ConnectionManagerInterface $connectionManager,
    private FieldTypeMapperInterface $fieldTypeMapper,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function createIndexTables(IndexInterface $index, array $backend_config): void {
    $index_id = (string) $index->id();

    // Create FTS5 virtual table.
    $this->createFts5Table($index_id, $index, $backend_config);

    // Create field data table using Drupal Schema API.
    $this->createFieldDataTable($index_id);

    // Create items tracking table.
    $this->createItemsTable($index_id);
  }

  /**
   * {@inheritdoc}
   */
  public function dropIndexTables(string $index_id): void {
    $connection = $this->connectionManager->getConnection($index_id);
    $schema = $connection->schema();

    // FTS5 table dropped via raw query (virtual table DDL).
    $fts_table = $this->getFtsTableName($index_id);
    $connection->query(sprintf('DROP TABLE IF EXISTS %s', $fts_table));

    // Use Drupal Schema API for regular tables.
    $field_data_table = $this->getFieldDataTableName($index_id);
    if ($schema->tableExists($field_data_table)) {
      $schema->dropTable($field_data_table);
    }

    $items_table = $this->getItemsTableName($index_id);
    if ($schema->tableExists($items_table)) {
      $schema->dropTable($items_table);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function updateIndexSchema(IndexInterface $index, array $backend_config): bool {
    $index_id = (string) $index->id();

    // For FTS5, any schema or configuration change requires recreating the
    // table and reindexing all content. Since we can't query the current
    // tokenizer configuration from FTS5, we simply always rebuild.
    // This is a safe approach as updateIndexSchema is only called when the
    // index configuration actually changes.
    $this->rebuildFts5Table($index_id, $index, $backend_config);

    // Always require reindex after schema update.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function tablesExist(string $index_id): bool {
    // Use Drupal's Schema API instead of raw sqlite_master query.
    // Check all three tables exist (FTS5, field_data, items).
    $connection = $this->connectionManager->getConnection($index_id);
    $schema = $connection->schema();

    // Check the FTS5 table and the regular tables.
    return $schema->tableExists($this->getFtsTableName($index_id))
      && $schema->tableExists($this->getFieldDataTableName($index_id))
      && $schema->tableExists($this->getItemsTableName($index_id));
  }

  /**
   * {@inheritdoc}
   */
  public function getFtsTableName(string $index_id): string {
    return $index_id . '_fts';
  }

  /**
   * {@inheritdoc}
   */
  public function getFieldDataTableName(string $index_id): string {
    return $index_id . '_field_data';
  }

  /**
   * {@inheritdoc}
   */
  public function getItemsTableName(string $index_id): string {
    return $index_id . '_items';
  }

  /**
   * Creates the FTS5 virtual table.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param array<string, mixed> $backend_config
   *   The backend configuration.
   */
  private function createFts5Table(string $index_id, IndexInterface $index, array $backend_config): void {
    $connection = $this->connectionManager->getConnection($index_id);
    $table_name = $this->getFtsTableName($index_id);

    // Build column definitions.
    $columns = $this->buildFts5Columns($index);

    // item_id is UNINDEXED (for filtering, not searching).
    $column_defs = ['item_id UNINDEXED'];
    foreach (array_keys($columns) as $column_name) {
      $column_defs[] = $column_name;
    }

    // Build tokenizer configuration string.
    $tokenizer_config = $this->buildTokenizerConfig($backend_config);

    // CREATE VIRTUAL TABLE requires raw query (not supported by Schema API).
    $sql = sprintf(
      'CREATE VIRTUAL TABLE IF NOT EXISTS %s USING fts5(%s, tokenize="%s")',
      $table_name,
      implode(', ', $column_defs),
      $tokenizer_config
    );

    $connection->query($sql);
  }

  /**
   * Creates the field data table for non-fulltext fields.
   *
   * @param string $index_id
   *   The Search API index ID.
   */
  private function createFieldDataTable(string $index_id): void {
    $connection = $this->connectionManager->getConnection($index_id);
    $table_name = $this->getFieldDataTableName($index_id);

    if ($connection->schema()->tableExists($table_name)) {
      return;
    }

    $schema = [
      'description' => 'Stores non-fulltext field values for filtering and faceting.',
      'fields' => [
        'id' => [
          'type' => 'serial',
          'unsigned' => TRUE,
          'not null' => TRUE,
        ],
        'item_id' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
        'field_name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
        'value_string' => [
          'type' => 'varchar',
          'length' => 1024,
          'not null' => FALSE,
        ],
        'value_integer' => [
          'type' => 'int',
          'size' => 'big',
          'not null' => FALSE,
        ],
        'value_decimal' => [
          'type' => 'float',
          'not null' => FALSE,
        ],
      ],
      'primary key' => ['id'],
      'indexes' => [
        // Index for delete/update operations by item.
        'item_id' => ['item_id'],
        // Covering indexes for condition filtering (field_name + value +
        // item_id). Including item_id enables index-only scans for subqueries.
        'field_value_string' => ['field_name', 'value_string', 'item_id'],
        'field_value_integer' => ['field_name', 'value_integer', 'item_id'],
        'field_value_decimal' => ['field_name', 'value_decimal', 'item_id'],
        // Index for facet aggregation and sort JOINs.
        'facet_lookup' => ['field_name', 'item_id'],
      ],
    ];

    $connection->schema()->createTable($table_name, $schema);
  }

  /**
   * Creates the items tracking table.
   *
   * @param string $index_id
   *   The Search API index ID.
   */
  private function createItemsTable(string $index_id): void {
    $connection = $this->connectionManager->getConnection($index_id);
    $table_name = $this->getItemsTableName($index_id);

    if ($connection->schema()->tableExists($table_name)) {
      return;
    }

    $schema = [
      'description' => 'Tracks indexed items metadata.',
      'fields' => [
        'item_id' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
        'datasource' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
        'language' => [
          'type' => 'varchar',
          'length' => 12,
          'not null' => FALSE,
        ],
        'indexed_at' => [
          'type' => 'int',
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['item_id'],
      'indexes' => [
        'datasource' => ['datasource'],
        'language' => ['language'],
        'indexed_at' => ['indexed_at'],
      ],
    ];

    $connection->schema()->createTable($table_name, $schema);
  }

  /**
   * Builds FTS5 column definitions from index fields.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   *
   * @return array<string, array<string, mixed>>
   *   Array of column definitions keyed by column name.
   */
  private function buildFts5Columns(IndexInterface $index): array {
    $columns = [];
    $fields = $index->getFields();

    foreach ($fields as $field_id => $field) {
      if ($this->fieldTypeMapper->isFulltextType($field->getType())) {
        // Sanitize field name for SQLite column.
        $column_name = ColumnNameHelper::sanitize($field_id);
        $columns[$column_name] = [
          'field_id' => $field_id,
          'type' => $field->getType(),
          'boost' => $field->getBoost(),
        ];
      }
    }

    return $columns;
  }

  /**
   * Rebuilds the FTS5 table with new schema.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param array<string, mixed> $backend_config
   *   The backend configuration.
   */
  private function rebuildFts5Table(string $index_id, IndexInterface $index, array $backend_config): void {
    $connection = $this->connectionManager->getConnection($index_id);
    $table_name = $this->getFtsTableName($index_id);
    $temp_table = $table_name . '_new';

    // Create new table with updated schema.
    $columns = $this->buildFts5Columns($index);
    $column_defs = ['item_id UNINDEXED'];
    foreach (array_keys($columns) as $column_name) {
      $column_defs[] = $column_name;
    }

    // Build tokenizer configuration string.
    $tokenizer_config = $this->buildTokenizerConfig($backend_config);

    // CREATE VIRTUAL TABLE requires raw query (not supported by Schema API).
    $connection->query(sprintf(
      'CREATE VIRTUAL TABLE %s USING fts5(%s, tokenize="%s")',
      $temp_table,
      implode(', ', $column_defs),
      $tokenizer_config
    ));

    // Drop old table and rename new one.
    $connection->query(sprintf('DROP TABLE IF EXISTS %s', $table_name));
    $connection->query(sprintf('ALTER TABLE %s RENAME TO %s', $temp_table, $table_name));
  }

  /**
   * Builds the FTS5 tokenizer configuration string.
   *
   * @param array<string, mixed> $backend_config
   *   The backend configuration.
   *
   * @return string
   *   The tokenizer configuration string for FTS5.
   */
  private function buildTokenizerConfig(array $backend_config): string {
    $tokenizer = Tokenizer::tryFrom($backend_config['tokenizer'] ?? '') ?? Tokenizer::Unicode61;

    // Trigram tokenizer has different options.
    if ($tokenizer === Tokenizer::Trigram) {
      $case_sensitive = $backend_config['trigram_case_sensitive'] ?? FALSE;
      if ($case_sensitive) {
        return 'trigram case_sensitive 1';
      }

      return 'trigram';
    }

    // For other tokenizers, just return the name.
    // Porter wraps unicode61 by default.
    return $tokenizer->value;
  }

}
