<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

use Drupal\search_api\IndexInterface;

/**
 * Defines the interface for managing SQLite database schema.
 */
interface SchemaManagerInterface {

  /**
   * Creates all tables for an index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param array<string, mixed> $backend_config
   *   The backend configuration.
   *
   * @throws \RuntimeException
   *   If table creation fails.
   */
  public function createIndexTables(IndexInterface $index, array $backend_config): void;

  /**
   * Drops all tables for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   */
  public function dropIndexTables(string $index_id): void;

  /**
   * Updates the schema when index fields change.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param array<string, mixed> $backend_config
   *   The backend configuration.
   *
   * @return bool
   *   TRUE if data needs to be reindexed, FALSE otherwise.
   */
  public function updateIndexSchema(IndexInterface $index, array $backend_config): bool;

  /**
   * Checks if the tables exist for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return bool
   *   TRUE if tables exist, FALSE otherwise.
   */
  public function tablesExist(string $index_id): bool;

  /**
   * Gets the FTS5 table name for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return string
   *   The FTS5 table name.
   */
  public function getFtsTableName(string $index_id): string;

  /**
   * Gets the field data table name for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return string
   *   The field data table name.
   */
  public function getFieldDataTableName(string $index_id): string;

  /**
   * Gets the items table name for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return string
   *   The items table name.
   */
  public function getItemsTableName(string $index_id): string;

}
