<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Index;

use Drupal\search_api\IndexInterface;

/**
 * Defines the interface for indexing items into SQLite.
 */
interface IndexerInterface {

  /**
   * Indexes items into the SQLite database.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param \Drupal\search_api\Item\ItemInterface[] $items
   *   The items to index, keyed by item ID.
   * @param array<string, mixed> $backendConfig
   *   The backend configuration.
   * @param callable|null $specialFieldsCallback
   *   Optional callback to get special fields for an item. The callback
   *   receives (IndexInterface $index, ItemInterface $item) and should return
   *   an array of FieldInterface objects keyed by field ID.
   *
   * @return array<int, string>
   *   Array of successfully indexed item IDs.
   *
   * @phpstan-ignore-next-line
   */
  public function indexItems(
    IndexInterface $index,
    array $items,
    array $backendConfig,
    ?callable $specialFieldsCallback = NULL,
  ): array;

  /**
   * Deletes items from the index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param array<string> $itemIds
   *   The item IDs to delete.
   */
  public function deleteItems(IndexInterface $index, array $itemIds): void;

  /**
   * Deletes all items from the index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param string|null $datasourceId
   *   Optional datasource ID to limit deletion.
   */
  public function deleteAllItems(IndexInterface $index, ?string $datasourceId = NULL): void;

  /**
   * Gets the number of indexed items.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   *
   * @return int
   *   The number of indexed items.
   */
  public function getIndexedItemsCount(IndexInterface $index): int;

}
