<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\search_api\Query\QueryInterface;

/**
 * Defines the interface for building facet queries.
 *
 * - AND facets use a temp table from the main query results (reused).
 * - OR facets rebuild the query excluding tagged condition groups.
 * - Supports min_count, limit, missing facets, and zero-count values.
 */
interface FacetBuilderInterface {

  /**
   * Calculates facets for a search query.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The Search API query (used for OR facet rebuilding).
   * @param \Drupal\Core\Database\Query\SelectInterface $db_query
   *   The database query (used for temp table creation for AND facets).
   * @param string $index_id
   *   The Search API index ID.
   * @param array<string, array<string, mixed>> $facet_options
   *   Facet definitions keyed by facet ID. Each facet has:
   *   - field: (string) Field name to facet on.
   *   - limit: (int) Max values to return (0 = unlimited).
   *   - min_count: (int) Min count threshold (0 = include zero-count values).
   *   - missing: (bool) Include count of items without a value.
   *   - operator: (string) 'and' or 'or'.
   * @param array<string, string> $fulltext_fields
   *   Map of field IDs to sanitized column names for fulltext fields.
   * @param int|null $result_count
   *   Total result count (for optimization - skip facets if < min_count).
   *
   * @return array<string, array<int, array{count: int, filter: string}>>
   *   Facet results keyed by facet ID. Each value is an array of:
   *   - count: (int) Number of items with this value.
   *   - filter: (string) Filter value in format '"value"' or '!' for missing.
   */
  public function getFacets(
    QueryInterface $query,
    SelectInterface $db_query,
    string $index_id,
    array $facet_options,
    array $fulltext_fields,
    ?int $result_count = NULL,
  ): array;

}
