<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;

/**
 * Defines the interface for building search queries.
 *
 * This interface handles all aspects of query building:
 * - FTS5 MATCH query construction.
 * - Condition handling (field_data subqueries).
 * - Sort handling (relevance, random, field-based).
 * - Pagination.
 */
interface QueryBuilderInterface {

  /**
   * Builds a complete search query from a Search API query.
   *
   * Returns a fully constructed SelectQuery that includes:
   * - FTS5 MATCH clause (if fulltext search)
   * - Condition filtering via field_data subqueries
   * - Sorting (relevance, random, or field-based)
   * - Pagination (range)
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The Search API query.
   * @param string $index_id
   *   The index ID.
   * @param array<string, string> $fulltext_fields
   *   Map of field IDs to sanitized column names for fulltext fields.
   * @param \Drupal\search_api_sqlite\Enum\MatchingMode $matching_mode
   *   The matching mode for fulltext search.
   * @param int $min_chars
   *   Minimum character length for search terms to be included.
   * @param \Drupal\search_api_sqlite\Enum\Tokenizer|null $tokenizer
   *   The tokenizer (required for Partial matching mode).
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   *   The constructed select query.
   */
  public function buildQuery(
    QueryInterface $query,
    string $index_id,
    array $fulltext_fields,
    MatchingMode $matching_mode = MatchingMode::Words,
    int $min_chars = 1,
    ?Tokenizer $tokenizer = NULL,
  ): SelectInterface;

  /**
   * Builds an FTS5 MATCH query string from search keys.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The Search API query.
   * @param array<string, string> $fulltext_fields
   *   Map of field IDs to sanitized column names.
   * @param \Drupal\search_api_sqlite\Enum\MatchingMode $matching_mode
   *   The matching mode.
   * @param int $min_chars
   *   Minimum character length for search terms to be included.
   * @param \Drupal\search_api_sqlite\Enum\Tokenizer|null $tokenizer
   *   The tokenizer (required for Partial matching mode).
   *
   * @return string|null
   *   The FTS5 MATCH query string, or NULL if no fulltext search.
   */
  public function buildMatchQuery(
    QueryInterface $query,
    array $fulltext_fields,
    MatchingMode $matching_mode = MatchingMode::Words,
    int $min_chars = 1,
    ?Tokenizer $tokenizer = NULL,
  ): ?string;

  /**
   * Escapes a search term for FTS5.
   *
   * @param string $term
   *   The search term.
   *
   * @return string
   *   The escaped term.
   */
  public function escapeTerm(string $term): string;

  /**
   * Parses user input into search terms and phrases.
   *
   * @param string $input
   *   The user's search input.
   * @param int $min_chars
   *   Minimum character length for terms to be included.
   *
   * @return array{terms: array<string>, phrases: array<string>}
   *   Array with 'terms' (individual words) and 'phrases' (quoted strings).
   */
  public function parseSearchInput(string $input, int $min_chars = 1): array;

}
