<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

/**
 * Defines the interface for query logging.
 *
 * Provides file-based query logging for debugging and performance analysis.
 * Logs are written to files alongside the SQLite database files.
 */
interface QueryLoggerInterface {

  /**
   * Enables query logging for a specific index.
   *
   * When enabled, all queries to the index's SQLite database will be logged
   * to a file in the same directory as the database file.
   *
   * @param string $index_id
   *   The Search API index ID.
   */
  public function enable(string $index_id): void;

  /**
   * Disables query logging for a specific index.
   *
   * @param string $index_id
   *   The Search API index ID.
   */
  public function disable(string $index_id): void;

  /**
   * Checks if query logging is enabled for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return bool
   *   TRUE if logging is enabled, FALSE otherwise.
   */
  public function isEnabled(string $index_id): bool;

  /**
   * Flushes any pending logged queries to the log file.
   *
   * This writes all accumulated queries to the log file and clears the buffer.
   * Called automatically on request termination.
   *
   * @param string|null $index_id
   *   The index ID to flush logs for, or NULL to flush all.
   */
  public function flush(?string $index_id = NULL): void;

  /**
   * Gets the log file path for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return string
   *   The absolute path to the log file.
   */
  public function getLogPath(string $index_id): string;

}
