<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Index;

use Drupal\search_api\IndexInterface;

/**
 * Interface for SQLite index maintenance operations.
 */
interface IndexOperationsInterface {

  /**
   * Gets statistics about an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return array<string, mixed>
   *   Array containing:
   *   - file_path: The SQLite database file path.
   *   - file_size: File size in bytes.
   *   - file_size_formatted: Human-readable file size.
   *   - indexed_items: Number of indexed items.
   *   - wal_file_exists: Whether WAL file exists.
   *   - wal_file_size: WAL file size in bytes.
   *   - tables: Array of table statistics.
   */
  public function getStatistics(string $index_id): array;

  /**
   * Optimizes the FTS5 index.
   *
   * Merges b-tree segments for improved query performance.
   * Safe to run anytime, recommended after heavy write operations.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @throws \RuntimeException
   *   If optimization fails.
   */
  public function optimize(string $index_id): void;

  /**
   * Vacuums the SQLite database.
   *
   * Reclaims disk space and defragments the database file.
   * May take time on large indexes.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @throws \RuntimeException
   *   If vacuum fails.
   */
  public function vacuum(string $index_id): void;

  /**
   * Checks the integrity of the FTS5 index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return array<string, mixed>
   *   Array containing:
   *   - valid: Whether the index is valid.
   *   - messages: Array of integrity check messages.
   */
  public function checkIntegrity(string $index_id): array;

  /**
   * Rebuilds the FTS5 index structures.
   *
   * Reconstructs the FTS5 shadow tables from the content.
   * Use if integrity check fails.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @throws \RuntimeException
   *   If rebuild fails.
   */
  public function rebuild(string $index_id): void;

  /**
   * Recreates the index by deleting and rebuilding the SQLite file.
   *
   * This is a destructive operation that deletes all indexed data.
   * The index will need to be reindexed after this operation.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   *
   * @throws \RuntimeException
   *   If recreation fails.
   */
  public function recreate(IndexInterface $index): void;

}
