<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Plugin\search_api\processor;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\search_api\Attribute\SearchApiProcessor;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Processor\ProcessorPluginBase;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api\Query\ResultSetInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Spellcheck\SpellCheckHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides spell check suggestions for search queries.
 *
 * When enabled and a search returns few results (below the configured
 * threshold), this processor checks the query for misspellings and suggests
 * corrections. The suggestions are stored in the query options for display
 * by Views area plugins (e.g., from search_api_spellcheck module).
 */
#[SearchApiProcessor(
  id: 'search_api_sqlite_spellcheck',
  label: new TranslatableMarkup('SQLite Spell Check'),
  description: new TranslatableMarkup('Suggests spelling corrections when search results are below threshold. Requires aspell, hunspell, or pspell.'),
  stages: [
    'postprocess_query' => 0,
  ],
)]
final class SpellCheck extends ProcessorPluginBase implements PluginFormInterface, ContainerFactoryPluginInterface {

  use PluginFormTrait;

  /**
   * Constructs a SpellCheck processor.
   *
   * @param array<string, mixed> $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\search_api_sqlite\Spellcheck\SpellCheckHandlerInterface $spellCheckHandler
   *   The spell check handler service.
   * @param \Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface $fts5QueryRunner
   *   The FTS5 query runner.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly SpellCheckHandlerInterface $spellCheckHandler,
    private readonly Fts5QueryRunnerInterface $fts5QueryRunner,
    private readonly SchemaManagerInterface $schemaManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   * @param array<string, mixed> $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('search_api_sqlite.spellcheck_handler'),
      $container->get('search_api_sqlite.fts5_query_runner'),
      $container->get('search_api_sqlite.schema_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function supportsIndex(IndexInterface $index): bool {
    $server = $index->getServerInstance();
    if ($server === NULL) {
      return FALSE;
    }

    // Only available for SQLite backend.
    if ($server->getBackendId() !== 'search_api_sqlite') {
      return FALSE;
    }

    // Check if backend supports spellcheck feature.
    return $server->supportsFeature('search_api_spellcheck');
  }

  /**
   * {@inheritdoc}
   *
   * @return array{backend: string, binary_path: string, language_mode: string, language: string, result_threshold: int, max_suggestions: int}
   *   The default configuration.
   */
  public function defaultConfiguration(): array {
    return [
      'backend' => 'aspell',
      'binary_path' => '',
      'language_mode' => 'auto',
      'language' => 'en',
      'result_threshold' => 5,
      'max_suggestions' => 3,
    ];
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The built form.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    // Check if the php-spellchecker library is installed.
    if (!$this->spellCheckHandler->isAvailable()) {
      $form['library_warning'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' .
        $this->t('The tigitz/php-spellchecker library is not installed. Run: <code>composer require tigitz/php-spellchecker</code>') .
        '</div>',
      ];

      return $form;
    }

    // Check for available backends (aspell/hunspell binaries, pspell).
    $available_backends = $this->spellCheckHandler->getAvailableBackends();

    if ($available_backends === []) {
      $form['warning'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--warning">' .
        $this->t('No spell check backends available. Install aspell, hunspell, or enable the pspell PHP extension.') .
        '</div>',
      ];
    }

    $form['info'] = [
      '#type' => 'details',
      '#title' => $this->t('How spell check works'),
      '#open' => FALSE,
    ];
    $form['info']['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('This processor uses external spell check tools (aspell, hunspell, or pspell) to detect misspellings in search queries and suggest corrections.') . '</p>' .
      '<p><strong>' . $this->t('How it works:') . '</strong></p>' .
      '<ul>' .
      '<li>' . $this->t('When a search returns few results (below threshold), the query is checked for misspellings.') . '</li>' .
      '<li>' . $this->t('Suggested corrections are validated against the search index to ensure they would return results.') . '</li>' .
      '<li>' . $this->t('Results are passed to Views area plugins (e.g., search_api_spellcheck module) for display.') . '</li>' .
      '</ul>' .
      '<p><strong>' . $this->t('Considerations:') . '</strong></p>' .
      '<ul>' .
      '<li>' . $this->t('Spell checking uses dictionary-based correction, not your indexed content. Domain-specific terms may be flagged as misspellings.') . '</li>' .
      '<li>' . $this->t('Each suggestion is validated against the FTS5 index, which adds some overhead. Lower "max suggestions" if experiencing issues.') . '</li>' .
      '<li>' . $this->t('The configured language should match your content language for best results.') . '</li>' .
      '<li>' . $this->t('Aspell/Hunspell require the corresponding language dictionaries to be installed on your server.') . '</li>' .
      '</ul>',
    ];

    $backend_options = [
      'aspell' => $this->t('Aspell'),
      'hunspell' => $this->t('Hunspell'),
      'pspell' => $this->t('PHP Pspell'),
    ];

    // Mark unavailable backends.
    foreach ($backend_options as $key => $label) {
      if (!in_array($key, $available_backends, TRUE)) {
        $backend_options[$key] = $this->t('@label (not available)', ['@label' => $label]);
      }
    }

    $form['backend'] = [
      '#type' => 'select',
      '#title' => $this->t('Spell check backend'),
      '#description' => $this->t('Select the spell checking library to use.'),
      '#options' => $backend_options,
      '#default_value' => $this->configuration['backend'],
      '#required' => TRUE,
    ];

    $form['binary_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Binary path'),
      '#description' => $this->t('Path to aspell/hunspell binary. Leave empty to use default system path.'),
      '#default_value' => $this->configuration['binary_path'],
      '#states' => [
        'visible' => [
          [':input[name="processors[search_api_sqlite_spellcheck][settings][backend]"]' => ['value' => 'aspell']],
          [':input[name="processors[search_api_sqlite_spellcheck][settings][backend]"]' => ['value' => 'hunspell']],
        ],
      ],
    ];

    $form['language_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Language mode'),
      '#description' => $this->t('How to determine the language for spell checking.'),
      '#options' => [
        'auto' => $this->t('Auto-detect from query'),
        'fixed' => $this->t('Use fixed language'),
      ],
      '#default_value' => $this->configuration['language_mode'],
      '#required' => TRUE,
    ];

    $form['language'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Language code'),
      '#description' => $this->t('Language code for spell checking (e.g., "en", "de", "fr"). Used as fallback in auto mode.'),
      '#default_value' => $this->configuration['language'],
      '#size' => 10,
      '#required' => TRUE,
    ];

    $form['result_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Result threshold'),
      '#description' => $this->t('Only suggest corrections when result count is below this threshold.'),
      '#default_value' => $this->configuration['result_threshold'],
      '#min' => 0,
      '#max' => 100,
      '#required' => TRUE,
    ];

    $form['max_suggestions'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum suggestions to validate'),
      '#description' => $this->t('Maximum number of spell check suggestions to validate against the search index. Only suggestions that would return actual results are shown. Lower values improve performance.'),
      '#default_value' => $this->configuration['max_suggestions'],
      '#min' => 1,
      '#max' => 10,
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->setConfiguration([
      'backend' => $form_state->getValue('backend'),
      'binary_path' => $form_state->getValue('binary_path') ?? '',
      'language_mode' => $form_state->getValue('language_mode'),
      'language' => $form_state->getValue('language'),
      'result_threshold' => (int) $form_state->getValue('result_threshold'),
      'max_suggestions' => (int) $form_state->getValue('max_suggestions'),
    ]);
  }

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\search_api\Query\ResultSetInterface<\Drupal\search_api\Item\ItemInterface> $results
   *   The search results.
   */
  public function postprocessSearchResults(ResultSetInterface $results): void {
    $query = $results->getQuery();

    // Only run if spell check was requested (can be TRUE or an array).
    $spellcheckOption = $query->getOption('search_api_spellcheck');
    if (empty($spellcheckOption)) {
      return;
    }

    // Check if spell check backend is available.
    $availableBackends = $this->spellCheckHandler->getAvailableBackends();
    if (!in_array($this->configuration['backend'], $availableBackends, TRUE)) {
      return;
    }

    // Only suggest if results are below threshold.
    if ($results->getResultCount() >= $this->configuration['result_threshold']) {
      return;
    }

    // Get the original search keys.
    $keys = $query->getOriginalKeys();
    if (empty($keys) || !is_string($keys)) {
      return;
    }

    // Determine language.
    $language = $this->getLanguage($query);

    // Build options for the spell check handler.
    $options = [];
    if (!empty($this->configuration['binary_path'])) {
      $options['binary_path'] = $this->configuration['binary_path'];
    }

    // Generate suggestions.
    $result = $this->spellCheckHandler->checkQuery(
      $keys,
      $this->configuration['backend'],
      $language,
      $options,
    );

    if (!$result['has_misspellings']) {
      return;
    }

    $index = $query->getIndex();

    // Validate collation against the FTS5 index.
    $collation = $result['collation'];
    if (!empty($collation)) {
      $validated = $this->validateSuggestions(
        [$collation],
        $index,
        1,
      );
      if ($validated === []) {
        // Collation doesn't match index, clear it.
        $collation = '';
      }
    }

    // Validate per-word suggestions against the FTS5 index.
    $validatedSuggestions = [];
    foreach ($result['suggestions'] as $word => $wordSuggestions) {
      $validatedWords = $this->validateSuggestions(
        $wordSuggestions,
        $index,
        $this->configuration['max_suggestions'],
      );
      if ($validatedWords !== []) {
        $validatedSuggestions[$word] = $validatedWords;
      }
    }

    // Build extra data for Views plugins.
    // DidYouMeanSpellCheck uses 'collation'.
    // SuggestionsSpellCheck uses 'suggestions' (per-word alternatives).
    $extraData = [];
    if (!empty($collation)) {
      $extraData['collation'] = $collation;
    }

    if ($validatedSuggestions !== []) {
      $extraData['suggestions'] = $validatedSuggestions;
    }

    if ($extraData !== []) {
      $results->setExtraData('search_api_spellcheck', $extraData);
    }
  }

  /**
   * Validates suggestions against the FTS5 index.
   *
   * Only returns suggestions that would actually yield search results.
   *
   * @param array<string> $suggestions
   *   The spell check suggestions to validate.
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param int $maxSuggestions
   *   Maximum number of suggestions to validate and return.
   *
   * @return array<string>
   *   Validated suggestions that exist in the index.
   */
  private function validateSuggestions(array $suggestions, IndexInterface $index, int $maxSuggestions): array {
    $indexId = $index->id();
    if ($indexId === NULL || !is_string($indexId)) {
      return [];
    }

    try {
      $ftsTable = $this->schemaManager->getFtsTableName($indexId);

      $validated = [];
      foreach ($suggestions as $suggestion) {
        if (count($validated) >= $maxSuggestions) {
          break;
        }

        // Escape and build FTS5 MATCH query for the suggestion.
        $matchQuery = $this->buildMatchQuery($suggestion);

        // Check if the suggestion would return results from the FTS5 index.
        if ($this->fts5QueryRunner->matchExists($indexId, $ftsTable, $matchQuery)) {
          $validated[] = $suggestion;
        }
      }

      return $validated;
    }
    catch (\Exception) {
      // On any error, return original suggestions.
      return array_slice($suggestions, 0, $maxSuggestions);
    }
  }

  /**
   * Builds an FTS5 MATCH query from a suggestion string.
   *
   * Uses prefix matching to find partial matches in the index.
   *
   * @param string $suggestion
   *   The suggestion to build a query for.
   *
   * @return string
   *   The FTS5 MATCH query string.
   */
  private function buildMatchQuery(string $suggestion): string {
    // Split into words and build prefix query for each.
    $words = preg_split('/\s+/', trim($suggestion), -1, PREG_SPLIT_NO_EMPTY);
    if ($words === FALSE || $words === []) {
      return '""';
    }

    // Use prefix matching (word*) for more flexible validation.
    // Quote each word to handle special characters, then add prefix operator.
    $escaped = array_map(
        // Escape double quotes within the word and add prefix operator.
        static fn(string $word): string => '"' . str_replace('"', '""', $word) . '"*', $words);

    // Use OR so any matching word validates the suggestion.
    return implode(' OR ', $escaped);
  }

  /**
   * Gets the language to use for spell checking.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The search query.
   *
   * @return string
   *   The language code.
   */
  private function getLanguage(QueryInterface $query): string {
    // If mode is 'fixed', use the configured language.
    if ($this->configuration['language_mode'] === 'fixed') {
      return $this->configuration['language'];
    }

    // Auto mode: try to detect from query languages.
    $languages = $query->getLanguages();
    if (!empty($languages)) {
      return reset($languages);
    }

    // Fall back to configured default.
    return $this->configuration['language'];
  }

}
