<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\search_api\Query\ConditionGroupInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;

/**
 * Builds complete search queries for FTS5 indexes.
 *
 * Handles all aspects of query building:
 * - FTS5 MATCH query construction based on MatchingMode.
 * - Condition handling via field_data subqueries.
 * - Sort handling (search_api_relevance, search_api_random, field-based).
 * - Pagination.
 */
final class QueryBuilder implements QueryBuilderInterface {

  /**
   * Characters that need escaping in FTS5.
   *
   * @var array<string>
   */
  private const array SPECIAL_CHARS = [
    '"', '*', '-', '+', 'AND', 'OR', 'NOT', 'NEAR', '(', ')', '.', ':', '^', '$',
  ];

  /**
   * Counter for unique join aliases.
   */
  private int $joinCounter = 0;

  /**
   * Counter for unique condition placeholders.
   */
  private int $placeholderCounter = 0;

  /**
   * Constructs a QueryBuilder instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Search\ConditionHelperInterface $conditionHelper
   *   The condition helper.
   */
  public function __construct(
    private readonly ConnectionManagerInterface $connectionManager,
    private readonly SchemaManagerInterface $schemaManager,
    private readonly ConditionHelperInterface $conditionHelper,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function buildQuery(
    QueryInterface $query,
    string $index_id,
    array $fulltext_fields,
    MatchingMode $matching_mode = MatchingMode::Words,
    int $min_chars = 1,
    ?Tokenizer $tokenizer = NULL,
    array $field_boosts = [],
  ): SelectInterface {
    // Reset counters for each query.
    $this->joinCounter = 0;
    $this->placeholderCounter = 0;

    $connection = $this->connectionManager->getConnection($index_id);
    $fts_table = $this->schemaManager->getFtsTableName($index_id);
    $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);
    $items_table = $this->schemaManager->getItemsTableName($index_id);

    // Get field definitions from index for proper type mapping.
    $index = $query->getIndex();
    $fields = $index->getFields();

    // Check if the entire search is negated at top level.
    $keys = $query->getKeys();
    $is_top_level_negation = is_array($keys) && !empty($keys['#negation']);

    // Build and apply FTS5 MATCH clause.
    $match_query = $this->buildMatchQuery($query, $fulltext_fields, $matching_mode, $min_chars, $tokenizer);
    $has_fulltext = $match_query !== NULL;

    if ($is_top_level_negation && $has_fulltext) {
      // Top-level negation: return all items EXCEPT those matching the query.
      // Start from items table (all items).
      $db_query = $connection->select($items_table, 'items');
      $db_query->fields('items', ['item_id']);
      $db_query->addExpression('0', 'score');

      // Build subquery for items that DO match the positive query.
      $match_subquery = $connection->select($fts_table, 'fts_neg');
      $match_subquery->fields('fts_neg', ['item_id']);
      $match_subquery->where($fts_table . ' MATCH :match_neg', [':match_neg' => $match_query]);

      // Exclude matching items.
      $db_query->condition('items.item_id', $match_subquery, 'NOT IN');
      // We're not using FTS for the main query now.
      $has_fulltext = FALSE;
    }
    elseif ($has_fulltext) {
      // Normal fulltext search.
      $db_query = $connection->select($fts_table, 'fts');
      $db_query->fields('fts', ['item_id']);
      // Add BM25 relevance score with field boosts.
      $bm25_expr = $this->buildBm25Expression($fts_table, $field_boosts);
      $db_query->addExpression($bm25_expr, 'score');
      // FTS5 MATCH requires table name, not alias.
      $db_query->where($fts_table . ' MATCH :match', [':match' => $match_query]);
    }
    else {
      // No fulltext search - query items table to get all items.
      $db_query = $connection->select($items_table, 'items');
      $db_query->fields('items', ['item_id']);
      $db_query->addExpression('0', 'score');
    }

    // Apply conditions from Search API query.
    $condition_group = $query->getConditionGroup();
    $this->applyConditionGroup(
      $db_query,
      $condition_group,
      $connection,
      $fts_table,
      $field_data_table,
      $fulltext_fields,
      $fields,
      $has_fulltext,
    );

    // Apply sorting.
    $this->applySorts($db_query, $query, $field_data_table, $has_fulltext);

    // Apply pagination.
    $options = $query->getOptions();
    $offset = $options['offset'] ?? 0;
    $limit = $query->getOption('limit') ?? 10;
    $db_query->range($offset, $limit);

    return $db_query;
  }

  /**
   * {@inheritdoc}
   */
  public function buildMatchQuery(
    QueryInterface $query,
    array $fulltext_fields,
    MatchingMode $matching_mode = MatchingMode::Words,
    int $min_chars = 1,
    ?Tokenizer $tokenizer = NULL,
  ): ?string {
    $keys = $query->getKeys();

    if (empty($keys)) {
      return NULL;
    }

    // Get the fulltext fields to search in.
    $search_fields = $query->getFulltextFields();
    if (empty($search_fields)) {
      $search_fields = array_keys($fulltext_fields);
    }

    // Map to sanitized column names.
    $columns = [];
    foreach ($search_fields as $field_id) {
      if (isset($fulltext_fields[$field_id])) {
        $columns[] = $fulltext_fields[$field_id];
      }
    }

    if ($columns === []) {
      return NULL;
    }

    // Parse the search keys.
    $parsed = $this->parseKeys($keys, $min_chars);

    if (empty($parsed['terms']) && empty($parsed['phrases']) && empty($parsed['groups'])) {
      return NULL;
    }

    // Build the FTS5 query.
    return $this->buildFts5Query($parsed, $columns, $matching_mode, $tokenizer);
  }

  /**
   * {@inheritdoc}
   */
  public function escapeTerm(string $term): string {
    $term = trim($term);

    // Double quotes need to be escaped by doubling them.
    $term = str_replace('"', '""', $term);

    // If term contains special chars or whitespace, quote it.
    $needs_quoting = FALSE;
    foreach (self::SPECIAL_CHARS as $char) {
      if (stripos($term, $char) !== FALSE) {
        $needs_quoting = TRUE;
        break;
      }
    }

    if ($needs_quoting || preg_match('/\s/', $term)) {
      return '"' . $term . '"';
    }

    return $term;
  }

  /**
   * {@inheritdoc}
   */
  public function parseSearchInput(string $input, int $min_chars = 1): array {
    $result = [
      'terms' => [],
      'phrases' => [],
    ];

    $input = trim($input);
    if ($input === '' || $input === '0') {
      return $result;
    }

    // Extract quoted phrases first.
    if (preg_match_all('/"([^"]+)"/', $input, $matches)) {
      $result['phrases'] = $matches[1];
      // Remove matched phrases from input.
      $input = preg_replace('/"[^"]+"/', '', $input) ?? '';
    }

    // Split remaining text into terms.
    $input = trim($input);
    if ($input !== '') {
      $terms = preg_split('/\s+/', $input, -1, PREG_SPLIT_NO_EMPTY);
      if ($terms !== FALSE) {
        // Filter out terms shorter than min_chars.
        $result['terms'] = array_values(
          array_filter($terms, fn(string $term): bool => mb_strlen($term) >= $min_chars)
        );
      }
    }

    return $result;
  }

  /**
   * Parses Search API keys into a structured format.
   *
   * Preserves nested groups with conjunction and negation information.
   *
   * @param mixed $keys
   *   The search keys from the query.
   * @param int $min_chars
   *   Minimum character length for terms to be included.
   *
   * @return array<string, mixed>
   *   Parsed search structure with keys:
   *   - terms: array of search terms
   *   - phrases: array of quoted phrases
   *   - groups: array of nested search groups
   *   - conjunction: 'AND' or 'OR'
   *   - negation: whether this group is negated
   */
  private function parseKeys(mixed $keys, int $min_chars = 1): array {
    $result = [
      'terms' => [],
      'phrases' => [],
      'groups' => [],
      'conjunction' => 'AND',
      'negation' => FALSE,
    ];

    if (is_string($keys)) {
      $parsed = $this->parseSearchInput($keys, $min_chars);
      return [
        'terms' => $parsed['terms'],
        'phrases' => $parsed['phrases'],
        'groups' => [],
        'conjunction' => 'AND',
        'negation' => FALSE,
      ];
    }

    if (!is_array($keys)) {
      return $result;
    }

    // Handle Search API's structured keys format.
    $conjunction = $keys['#conjunction'] ?? 'AND';
    $result['conjunction'] = strtoupper((string) $conjunction);
    $result['negation'] = !empty($keys['#negation']);

    foreach ($keys as $key => $value) {
      if (str_starts_with((string) $key, '#')) {
        continue;
      }

      if (is_array($value)) {
        // Nested group - recursively parse and store as a group.
        $nested = $this->parseKeys($value, $min_chars);
        if (!empty($nested['terms']) || !empty($nested['phrases']) || !empty($nested['groups'])) {
          $result['groups'][] = $nested;
        }
      }
      else {
        // Single term.
        $term = trim((string) $value);
        if (mb_strlen($term) >= $min_chars) {
          $result['terms'][] = $term;
        }
      }
    }

    return $result;
  }

  /**
   * Builds the final FTS5 query string.
   *
   * @param array $parsed
   *   Parsed search structure from parseKeys().
   * @param array $columns
   *   Column names to search in.
   * @param \Drupal\search_api_sqlite\Enum\MatchingMode $matching_mode
   *   The matching mode.
   * @param \Drupal\search_api_sqlite\Enum\Tokenizer|null $tokenizer
   *   The tokenizer (for Partial mode behavior).
   * @param bool $is_nested
   *   Whether this is a nested call (skip column filter).
   *
   * @return string
   *   The FTS5 MATCH query.
   */
  private function buildFts5Query(
    array $parsed,
    array $columns,
    MatchingMode $matching_mode,
    ?Tokenizer $tokenizer = NULL,
    bool $is_nested = FALSE,
  ): string {
    $positive_parts = [];
    $negative_parts = [];
    // FTS5 requires explicit AND when mixing with parenthesized groups.
    $conjunction = $parsed['conjunction'] === 'OR' ? ' OR ' : ' AND ';

    // Build column filter only for top-level query, not nested groups.
    $column_filter = (!$is_nested && $columns !== []) ? '{' . implode(' ', $columns) . '}: ' : '';

    // Process phrases (exact matches).
    foreach ($parsed['phrases'] as $phrase) {
      $escaped = $this->escapeTerm($phrase);
      $positive_parts[] = $escaped;
    }

    // In Phrase mode, combine all terms into a single quoted phrase.
    if ($matching_mode === MatchingMode::Phrase && !empty($parsed['terms'])) {
      $phrase_terms = implode(' ', $parsed['terms']);
      $positive_parts[] = '"' . $phrase_terms . '"';
    }
    else {
      // Process individual terms.
      foreach ($parsed['terms'] as $term) {
        $escaped = $this->escapeTerm($term);

        // Handle prefix and partial matching differently.
        if ($matching_mode === MatchingMode::Prefix && !str_ends_with($escaped, '*')) {
          // Prefix matching: always add * suffix.
          $escaped = trim($escaped, '"');
          $escaped .= '*';
        }
        elseif (
          $matching_mode === MatchingMode::Partial
          && !str_ends_with($escaped, '*')
        ) {
          // Partial matching:
          // - Trigram tokenizer: search term as-is (substring search).
          // - Other tokenizers: fall back to prefix matching.
          if ($tokenizer !== Tokenizer::Trigram) {
            $escaped = trim($escaped, '"');
            $escaped .= '*';
          }

          // else: keep escaped term as-is for trigram substring search.
        }

        $positive_parts[] = $escaped;
      }
    }

    // Process nested groups recursively.
    foreach ($parsed['groups'] as $group) {
      $group_query = $this->buildFts5Query($group, $columns, $matching_mode, $tokenizer, TRUE);
      if ($group_query !== '') {
        // Separate positive and negative groups.
        if (!empty($group['negation'])) {
          $negative_parts[] = '(' . $group_query . ')';
        }
        else {
          $positive_parts[] = '(' . $group_query . ')';
        }
      }
    }

    if ($positive_parts === [] && $negative_parts === []) {
      return '';
    }

    // Build the query: positive parts joined by conjunction.
    $query = implode($conjunction, $positive_parts);

    // Append negative parts with NOT (no AND before NOT).
    foreach ($negative_parts as $neg) {
      if ($query !== '') {
        $query .= ' NOT ' . $neg;
      }
    }

    // Wrap entire query with column filter at top level only.
    if ($column_filter !== '' && !$is_nested) {
      return $column_filter . '(' . $query . ')';
    }

    return $query;
  }

  /**
   * Applies a condition group to the query.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $db_query
   *   The database query.
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   The condition group.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $fts_table
   *   The FTS5 table name.
   * @param string $field_data_table
   *   The field data table name.
   * @param array $fulltext_fields
   *   Map of fulltext field IDs to sanitized column names.
   * @param array<\Drupal\search_api\Item\FieldInterface> $fields
   *   The index field definitions.
   * @param bool $has_fulltext
   *   Whether this is a fulltext query (uses 'fts' alias) or not ('items').
   */
  private function applyConditionGroup(
    SelectInterface $db_query,
    ConditionGroupInterface $condition_group,
    $connection,
    string $fts_table,
    string $field_data_table,
    array $fulltext_fields,
    array $fields,
    bool $has_fulltext = TRUE,
  ): void {
    $conditions = $condition_group->getConditions();
    if (empty($conditions)) {
      return;
    }

    // Determine the correct table alias for item_id.
    $item_id_column = $has_fulltext ? 'fts.item_id' : 'items.item_id';
    $conjunction = $condition_group->getConjunction();

    // Collect subqueries for this group.
    $subqueries = [];

    foreach ($conditions as $condition) {
      if ($condition instanceof ConditionGroupInterface) {
        // Nested condition group - handle based on parent conjunction.
        if ($conjunction === 'OR') {
          // For OR at this level, nested groups become single subqueries.
          // Build a subquery that handles the nested group's conditions.
          $nested_subquery = $this->buildNestedGroupSubquery(
            $condition,
            $connection,
            $fts_table,
            $field_data_table,
            $fulltext_fields,
            $fields,
          );
          if ($nested_subquery instanceof SelectInterface) {
            $subqueries[] = $nested_subquery;
          }
        }
        else {
          // For AND at this level, apply nested group directly to main query.
          $this->applyConditionGroup(
            $db_query,
            $condition,
            $connection,
            $fts_table,
            $field_data_table,
            $fulltext_fields,
            $fields,
            $has_fulltext,
          );
        }
      }
      else {
        // Simple condition - build subquery.
        $field = $condition->getField();
        $value = $condition->getValue();
        $operator = $condition->getOperator();

        $result = $this->buildConditionSubquery(
          $connection,
          $fts_table,
          $field_data_table,
          $fulltext_fields,
          $fields,
          $field,
          $value,
          $operator,
        );

        if ($result !== NULL) {
          [$subquery, $outer_operator] = $result;
          if ($conjunction === 'OR') {
            // Collect for UNION.
            // NOT IN conditions with OR conjunction need special handling.
            $subqueries[] = $subquery;
          }
          else {
            // AND - apply directly as separate condition.
            $db_query->condition($item_id_column, $subquery, $outer_operator);
          }
        }
      }
    }

    // For OR conjunction, combine all subqueries with UNION.
    if ($conjunction === 'OR' && $subqueries !== []) {
      $combined = $this->combineSubqueriesWithUnion($subqueries);
      if ($combined instanceof SelectInterface) {
        $db_query->condition($item_id_column, $combined, 'IN');
      }
    }
  }

  /**
   * Builds a subquery for a nested condition group.
   *
   * For nested AND groups, returns items matching ALL conditions.
   * For nested OR groups, returns items matching ANY condition.
   *
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   The nested condition group.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $fts_table
   *   The FTS5 table name.
   * @param string $field_data_table
   *   The field data table name.
   * @param array $fulltext_fields
   *   Map of fulltext field IDs to sanitized column names.
   * @param array $fields
   *   The index field definitions.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface|null
   *   A subquery returning item_ids, or NULL if empty.
   */
  private function buildNestedGroupSubquery(
    ConditionGroupInterface $condition_group,
    $connection,
    string $fts_table,
    string $field_data_table,
    array $fulltext_fields,
    array $fields,
  ): ?SelectInterface {
    $nested_conjunction = $condition_group->getConjunction();
    $conditions = $condition_group->getConditions();

    if (empty($conditions)) {
      return NULL;
    }

    // Collect all subqueries from this group.
    $subqueries = [];
    foreach ($conditions as $condition) {
      if ($condition instanceof ConditionGroupInterface) {
        // Recursively handle deeply nested groups.
        $nested = $this->buildNestedGroupSubquery(
          $condition,
          $connection,
          $fts_table,
          $field_data_table,
          $fulltext_fields,
          $fields,
        );
        if ($nested instanceof SelectInterface) {
          $subqueries[] = $nested;
        }
      }
      else {
        $result = $this->buildConditionSubquery(
          $connection,
          $fts_table,
          $field_data_table,
          $fulltext_fields,
          $fields,
          $condition->getField(),
          $condition->getValue(),
          $condition->getOperator(),
        );
        if ($result !== NULL) {
          $subqueries[] = $result[0];
        }
      }
    }

    if ($subqueries === []) {
      return NULL;
    }

    if (count($subqueries) === 1) {
      return $subqueries[0];
    }

    if ($nested_conjunction === 'OR') {
      // OR: UNION all subqueries.
      return $this->combineSubqueriesWithUnion($subqueries);
    }

    // AND: INTERSECT all subqueries - items must be in ALL subqueries.
    return $this->combineSubqueriesWithIntersect($subqueries, $connection);
  }

  /**
   * Combines subqueries with INTERSECT (items must be in ALL subqueries).
   *
   * Since Drupal's DB API doesn't support INTERSECT directly, we simulate it
   * by nesting IN conditions: items must be IN subquery1 AND IN subquery2...
   *
   * @param array $subqueries
   *   Array of subqueries.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface|null
   *   Combined query, or NULL if empty.
   */
  private function combineSubqueriesWithIntersect(
    array $subqueries,
    $connection,
  ): ?SelectInterface {
    if ($subqueries === []) {
      return NULL;
    }

    if (count($subqueries) === 1) {
      return $subqueries[0];
    }

    // Start with first subquery as base.
    $first = array_shift($subqueries);

    // Build a wrapper query that requires item_id to be in all subqueries.
    // item_id IN (second) AND item_id IN (third)...
    $wrapper = $connection->select($first, 'base');
    $wrapper->addField('base', 'item_id');

    foreach ($subqueries as $subquery) {
      $wrapper->condition('base.item_id', $subquery, 'IN');
    }

    return $wrapper;
  }

  /**
   * Combines multiple subqueries with UNION.
   *
   * @param array $subqueries
   *   Array of subqueries.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface|null
   *   Combined query, or NULL if empty.
   */
  private function combineSubqueriesWithUnion(
    array $subqueries,
  ): ?SelectInterface {
    if ($subqueries === []) {
      return NULL;
    }

    if (count($subqueries) === 1) {
      return reset($subqueries);
    }

    // Use the first subquery as base and add UNION for the rest.
    $base = array_shift($subqueries);
    foreach ($subqueries as $subquery) {
      $base->union($subquery);
    }

    return $base;
  }

  /**
   * Builds a subquery for a single condition.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $fts_table
   *   The FTS5 table name.
   * @param string $field_data_table
   *   The field data table name.
   * @param array $fulltext_fields
   *   Map of fulltext field IDs to sanitized column names.
   * @param array<\Drupal\search_api\Item\FieldInterface> $fields
   *   The index field definitions.
   * @param string $field
   *   The field name.
   * @param mixed $value
   *   The condition value.
   * @param string $operator
   *   The condition operator.
   *
   * @return array|null
   *   Array of [subquery, outer_operator], or NULL if cannot be applied.
   */
  private function buildConditionSubquery(
    $connection,
    string $fts_table,
    string $field_data_table,
    array $fulltext_fields,
    array $fields,
    string $field,
    mixed $value,
    string $operator,
  ): ?array {
    // Handle search_api_language specially - stored in items table.
    if ($field === 'search_api_language') {
      $items_table = $this->schemaManager->getItemsTableName(
        $this->extractIndexIdFromFtsTable($fts_table)
      );
      $subquery = $connection->select($items_table, 'it');
      $subquery->fields('it', ['item_id']);

      $outer_operator = 'IN';
      switch (strtoupper($operator)) {
        case 'IN':
          $subquery->condition('it.language', $value, 'IN');
          break;

        case 'NOT IN':
          $subquery->condition('it.language', $value, 'IN');
          $outer_operator = 'NOT IN';
          break;

        case '=':
          $subquery->condition('it.language', $value);
          break;

        case '<>':
        case '!=':
          $subquery->condition('it.language', $value);
          $outer_operator = 'NOT IN';
          break;
      }

      return [$subquery, $outer_operator];
    }

    // Handle search_api_id specially - filter by item_id directly.
    if ($field === 'search_api_id') {
      $items_table = $this->schemaManager->getItemsTableName(
      $this->extractIndexIdFromFtsTable($fts_table)
      );
      $subquery = $connection->select($items_table, 'it');
      $subquery->fields('it', ['item_id']);

      $outer_operator = 'IN';
      switch (strtoupper($operator)) {
        case 'IN':
          $subquery->condition('it.item_id', $value, 'IN');
          break;

        case 'NOT IN':
          $subquery->condition('it.item_id', $value, 'IN');
          $outer_operator = 'NOT IN';
          break;

        case '<>':
        case '!=':
          $subquery->condition('it.item_id', $value);
          $outer_operator = 'NOT IN';
          break;

        case '>':
        case '>=':
        case '<':
        case '<=':
        case '=':
        default:
          $subquery->condition('it.item_id', $value, $operator);
          break;
      }

      return [$subquery, $outer_operator];
    }

    // Handle search_api_datasource specially - stored in items table.
    if ($field === 'search_api_datasource') {
      $items_table = $this->schemaManager->getItemsTableName(
        $this->extractIndexIdFromFtsTable($fts_table)
      );
      $subquery = $connection->select($items_table, 'it');
      $subquery->fields('it', ['item_id']);

      $outer_operator = 'IN';
      switch (strtoupper($operator)) {
        case 'IN':
          $subquery->condition('it.datasource', $value, 'IN');
          break;

        case 'NOT IN':
          $subquery->condition('it.datasource', $value, 'IN');
          $outer_operator = 'NOT IN';
          break;

        case '=':
          $subquery->condition('it.datasource', $value);
          break;

        case '<>':
        case '!=':
          $subquery->condition('it.datasource', $value);
          $outer_operator = 'NOT IN';
          break;
      }

      return [$subquery, $outer_operator];
    }

    // Skip other special search API fields.
    if (str_starts_with($field, 'search_api_')) {
      return NULL;
    }

    // Check if this is a fulltext field - use FTS5 MATCH for text conditions.
    if (isset($fulltext_fields[$field]) && is_string($value) && in_array(strtoupper($operator), ['=', '<>'], TRUE)) {
      $subquery = $this->buildFulltextConditionSubquery(
        $connection,
        $fts_table,
        $fulltext_fields[$field],
        $value,
      );
      // For '<>' on fulltext, use NOT IN on the outer query.
      $outer_op = strtoupper($operator) === '<>' ? 'NOT IN' : 'IN';
      return [$subquery, $outer_op];
    }

    // Handle NULL value conditions on fulltext fields.
    // For FTS5, we check if the column has any content (non-empty string).
    if (isset($fulltext_fields[$field]) && $value === NULL) {
      $column_name = $fulltext_fields[$field];
      // Find items where the FTS5 column is NOT empty.
      $items_with_content = $connection->select($fts_table, 'fts_null');
      $items_with_content->fields('fts_null', ['item_id']);
      $items_with_content->where(sprintf("fts_null.%s != ''", $column_name));

      if (in_array(strtoupper($operator), ['<>', '!='], TRUE)) {
        // Name <> NULL: find items that HAVE content in the FTS field.
        return [$items_with_content, 'IN'];
      }

      // Name = NULL: find items WITHOUT content in the FTS field.
      return [$items_with_content, 'NOT IN'];
    }

    // Handle NULL value conditions specially.
    // When value is NULL, we need to find items that don't have any row
    // in field_data for this field (or do have a row, for <> NULL).
    if ($value === NULL) {
      // Get all items that HAVE a value for this field.
      $items_with_field = $connection->select($field_data_table, 'fd');
      $items_with_field->fields('fd', ['item_id']);
      $items_with_field->condition('fd.field_name', $field);
      $items_with_field->distinct();

      if (in_array(strtoupper($operator), ['<>', '!='], TRUE)) {
        // Category <> NULL: find items that HAVE the field.
        return [$items_with_field, 'IN'];
      }

      // Category = NULL (or IS NULL): find items WITHOUT the field.
      // Return the subquery of items WITH the field, but use NOT IN.
      return [$items_with_field, 'NOT IN'];
    }

    // Non-fulltext field - query field_data table.
    $subquery = $connection->select($field_data_table, 'fd');
    $subquery->fields('fd', ['item_id']);
    $subquery->condition('fd.field_name', $field);

    // Determine value column based on field type or value type.
    $value_column = $this->conditionHelper->getValueColumnForField($field, $fields, $value);

    // Default outer operator.
    $outer_operator = 'IN';

    // Handle different operators.
    switch (strtoupper($operator)) {
      case 'IN':
        if (is_array($value)) {
          // Find items that HAVE any of these values.
          $subquery->condition('fd.' . $value_column, $value, 'IN');
        }

        break;

      case 'NOT IN':
        if (is_array($value)) {
          // Find items that HAVE any of these values, then exclude them.
          // Build subquery to find items with ANY of the excluded values.
          $subquery->condition('fd.' . $value_column, $value, 'IN');
          $outer_operator = 'NOT IN';
        }

        break;

      case 'BETWEEN':
        if (is_array($value) && count($value) >= 2) {
          $subquery->condition('fd.' . $value_column, $value, 'BETWEEN');
        }

        break;

      case 'NOT BETWEEN':
        if (is_array($value) && count($value) >= 2) {
          // Find items in range, then exclude them.
          $subquery->condition('fd.' . $value_column, $value, 'BETWEEN');
          $outer_operator = 'NOT IN';
        }

        break;

      case 'IS NULL':
        $subquery->isNull('fd.' . $value_column);
        break;

      case 'IS NOT NULL':
        $subquery->isNotNull('fd.' . $value_column);
        break;

      case '<>':
      case '!=':
        // Find items that have this value, then exclude them.
        $subquery->condition('fd.' . $value_column, $value, '=');
        $outer_operator = 'NOT IN';
        break;

      case '>':
      case '>=':
      case '<':
      case '<=':
      case '=':
      default:
        $subquery->condition('fd.' . $value_column, $value, $operator);
        break;
    }

    return [$subquery, $outer_operator];
  }

  /**
   * Builds a subquery for a fulltext field condition using FTS5 MATCH.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $fts_table
   *   The FTS5 table name.
   * @param string $column_name
   *   The sanitized column name in the FTS5 table.
   * @param string $value
   *   The search value.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   *   The subquery.
   */
  private function buildFulltextConditionSubquery(
    $connection,
    string $fts_table,
    string $column_name,
    string $value,
  ): SelectInterface {
    $escaped = $this->escapeTerm($value);
    $match_query = '{' . $column_name . '}: ' . $escaped;

    // Use unique placeholder to avoid conflicts when multiple subqueries
    // are combined with UNION.
    $placeholder = ':fts_match_' . $this->placeholderCounter++;

    $subquery = $connection->select($fts_table, 'fts_cond');
    $subquery->fields('fts_cond', ['item_id']);
    $subquery->where(sprintf('%s MATCH %s', $fts_table, $placeholder), [$placeholder => $match_query]);

    // Note: For '<>' (not contains), we would need to return all items NOT in
    // this subquery. For now, we only support '=' (contains) for fulltext.
    // The caller should handle '<>' by using NOT IN instead of IN.
    return $subquery;
  }

  /**
   * Applies sorting to the query.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $db_query
   *   The database query.
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The Search API query.
   * @param string $field_data_table
   *   The field data table name.
   * @param bool $has_fulltext
   *   Whether this is a fulltext query (uses 'fts' alias) or not ('items').
   */
  private function applySorts(
    SelectInterface $db_query,
    QueryInterface $query,
    string $field_data_table,
    bool $has_fulltext = TRUE,
  ): void {
    /** @var array<string, string> $sorts */
    $sorts = $query->getSorts();

    // Default sort by relevance if no sorts specified and has fulltext search.
    if (empty($sorts)) {
      $db_query->orderBy('score', 'DESC');
      return;
    }

    // Determine the table alias for item_id.
    $table_alias = $has_fulltext ? 'fts' : 'items';

    foreach ($sorts as $field => $direction) {
      $direction = strtoupper($direction) === 'ASC' ? 'ASC' : 'DESC';

      match ($field) {
        'search_api_relevance' => $db_query->orderBy('score', $direction),
        'search_api_random' => $db_query->orderRandom(),
        'search_api_id' => $db_query->orderBy($table_alias . '.item_id', $direction),
        default => $this->applyFieldSort(
          $db_query,
          $field,
          $direction,
          $field_data_table,
          $table_alias,
        ),
      };
    }
  }

  /**
   * Applies a field-based sort via LEFT JOIN to field_data.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $db_query
   *   The database query.
   * @param string $field
   *   The field name to sort by.
   * @param string $direction
   *   The sort direction (ASC or DESC).
   * @param string $field_data_table
   *   The field data table name.
   * @param string $table_alias
   *   The main table alias ('fts' or 'items').
   */
  private function applyFieldSort(
    SelectInterface $db_query,
    string $field,
    string $direction,
    string $field_data_table,
    string $table_alias,
  ): void {
    // Create unique alias for this sort join.
    $alias = 'sort_' . $this->joinCounter++;

    // Join field_data table for the sort field value.
    // Using leftJoin to include items without this field value.
    // Must qualify item_id with table alias to avoid ambiguity.
    $db_query->leftJoin(
      $field_data_table,
      $alias,
      sprintf('%s.item_id = %s.item_id AND %s.field_name = :sort_field_%s', $table_alias, $alias, $alias, $alias),
      [':sort_field_' . $alias => $field],
    );

    // Order by appropriate value column.
    // We try integer first, then decimal, then string.
    // This could be optimized by knowing the field type from schema.
    $db_query->addExpression(
      sprintf('COALESCE(%s.value_integer, %s.value_decimal, %s.value_string)', $alias, $alias, $alias),
      'sort_value_' . $alias
    );
    $db_query->orderBy('sort_value_' . $alias, $direction);
  }

  /**
   * Extracts the index ID from an FTS table name.
   *
   * @param string $fts_table
   *   The FTS table name (e.g., 'my_index_fts').
   *
   * @return string
   *   The index ID.
   */
  private function extractIndexIdFromFtsTable(string $fts_table): string {
    // Remove the '_fts' suffix.
    return preg_replace('/_fts$/', '', $fts_table) ?? $fts_table;
  }

  /**
   * Builds the BM25 expression with optional field boosts.
   *
   * FTS5's bm25() function accepts optional weight parameters for each column
   * in the order they were defined. Higher weights increase the relevance
   * contribution of matches in that column.
   *
   * @param string $fts_table
   *   The FTS5 table name.
   * @param array<float> $boosts
   *   Array of boost values in column order.
   *
   * @return string
   *   The BM25 SQL expression.
   */
  private function buildBm25Expression(string $fts_table, array $boosts): string {
    if ($boosts === []) {
      return sprintf('bm25(%s)', $fts_table);
    }

    // Format boost values to avoid floating point issues.
    $boost_args = array_map(
      fn(float $boost): string => number_format($boost, 2, '.', ''),
      $boosts
    );

    return sprintf('bm25(%s, %s)', $fts_table, implode(', ', $boost_args));
  }

}
