<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Spellcheck;

use PhpSpellcheck\Spellchecker\Aspell;
use PhpSpellcheck\Spellchecker\Hunspell;
use PhpSpellcheck\Spellchecker\PHPPspell;
use Psr\Log\LoggerInterface;

/**
 * Service for handling spell checking.
 */
final readonly class SpellCheckHandler implements SpellCheckHandlerInterface {

  /**
   * Constructs a SpellCheckHandler instance.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    return class_exists(Aspell::class);
  }

  /**
   * {@inheritdoc}
   */
  public function checkQuery(string $query, string $backend, string $language, array $options = []): array {
    if (!$this->isAvailable()) {
      $this->logger->debug('php-spellchecker library not available');
      return $this->emptyResult();
    }

    try {
      // Get binary path from options (empty string means use default).
      $binaryPath = empty($options['binary_path']) ? NULL : $options['binary_path'];

      // Create spellchecker based on backend.
      $spellchecker = match ($backend) {
        'aspell' => Aspell::create($binaryPath),
        'hunspell' => Hunspell::create($binaryPath),
        'pspell' => new PHPPspell(),
        default => throw new \RuntimeException('Unknown backend: ' . $backend),
      };

      // Check for misspellings.
      $misspellings = $spellchecker->check($query, [$language]);

      // Build per-word suggestions from misspellings.
      // Store up to 5 alternatives per misspelled word.
      $perWordSuggestions = [];
      $corrections = [];
      foreach ($misspellings as $misspelling) {
        $word = $misspelling->getWord();
        $wordSuggestions = $misspelling->getSuggestions();
        if (!empty($wordSuggestions)) {
          // Store multiple suggestions per word (up to 5).
          $perWordSuggestions[$word] = array_slice($wordSuggestions, 0, 5);
          // Use first suggestion for collation.
          $corrections[$word] = $wordSuggestions[0];
        }
      }

      // No misspellings found.
      if ($corrections === []) {
        return $this->emptyResult();
      }

      // Generate collation (best corrected query using first suggestion each).
      $collation = $query;
      foreach ($corrections as $word => $replacement) {
        $collation = str_replace($word, $replacement, $collation);
      }

      return [
        'has_misspellings' => TRUE,
        'collation' => $collation !== $query ? $collation : '',
        'suggestions' => $perWordSuggestions,
      ];

    }
    catch (\Exception $exception) {
      $this->logger->warning('Spell check failed: @message', [
        '@message' => $exception->getMessage(),
      ]);
      return $this->emptyResult();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableBackends(): array {
    if (!$this->isAvailable()) {
      return [];
    }

    $available = [];

    foreach (['aspell', 'hunspell', 'pspell'] as $backend) {
      if ($this->testBackend($backend)) {
        $available[] = $backend;
      }
    }

    return $available;
  }

  /**
   * Tests if a backend is available.
   */
  private function testBackend(string $backend): bool {
    try {
      $spellchecker = match ($backend) {
        'aspell' => Aspell::create(),
        'hunspell' => Hunspell::create(),
        'pspell' => new PHPPspell(),
        default => NULL,
      };

      if ($spellchecker === NULL) {
        return FALSE;
      }

      // Test with simple word.
      $spellchecker->check('test', ['en']);
      return TRUE;
    }
    catch (\Exception) {
      return FALSE;
    }
  }

  /**
   * Returns empty result structure.
   *
   * @return array{has_misspellings: bool, collation: string, suggestions: array<string, array<string>>}
   *   Empty result array.
   */
  private function emptyResult(): array {
    return [
      'has_misspellings' => FALSE,
      'collation' => '',
      'suggestions' => [],
    ];
  }

}
