<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Status;

use Drupal\search_api\ServerInterface;

/**
 * Interface for the status reporter service.
 *
 * This service is responsible for gathering and formatting status information
 * about the SQLite FTS5 backend configuration and runtime state.
 */
interface StatusReporterInterface {

  /**
   * Gets the view settings array for a backend configuration.
   *
   * @param array<string, mixed> $configuration
   *   The backend configuration.
   * @param \Drupal\search_api\ServerInterface $server
   *   The search server.
   *
   * @return array<int, array<string, mixed>>
   *   An array of settings info items, each with 'label' and 'info' keys.
   */
  public function getViewSettings(array $configuration, ServerInterface $server): array;

  /**
   * Gets SQLite version information.
   *
   * @return array{version: string, fts5_available: bool}
   *   Array with version string and FTS5 availability.
   */
  public function getSqliteVersionInfo(): array;

  /**
   * Gets statistics for a specific index.
   *
   * @param string $index_id
   *   The index ID.
   *
   * @return array{database_exists: bool, tables_exist: bool, file_size: int, document_count: int, filename: string}
   *   Array of statistics.
   */
  public function getIndexStatistics(string $index_id): array;

  /**
   * Gets detailed statistics for an index including WAL and table info.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return array<string, mixed>
   *   Array containing:
   *   - file_path: The SQLite database file path.
   *   - file_size: File size in bytes.
   *   - file_size_formatted: Human-readable file size.
   *   - indexed_items: Number of indexed items.
   *   - wal_file_exists: Whether WAL file exists.
   *   - wal_file_size: WAL file size in bytes.
   *   - wal_file_size_formatted: Human-readable WAL file size.
   *   - tables: Array of table statistics.
   */
  public function getDetailedStatistics(string $index_id): array;

}
