<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Database;

use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\FieldInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Schema;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManager;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Unit tests for SchemaManager table name generation and simple logic.
 *
 * Database operations (createIndexTables, dropIndexTables, updateIndexSchema)
 * are tested in Kernel tests with a real SQLite connection.
 */
#[CoversClass(SchemaManager::class)]
#[Group('search_api_sqlite')]
final class SchemaManagerTest extends UnitTestCase {

  /**
   * The schema manager under test.
   */
  private SchemaManager $schemaManager;

  /**
   * The mocked connection manager.
   */
  private ConnectionManagerInterface&MockObject $connectionManager;

  /**
   * The mocked field type mapper.
   */
  private FieldTypeMapperInterface&MockObject $fieldTypeMapper;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->fieldTypeMapper = $this->createMock(FieldTypeMapperInterface::class);

    $this->schemaManager = new SchemaManager(
      $this->connectionManager,
      $this->fieldTypeMapper
    );
  }

  /**
   * Tests getFtsTableName() generates correct table name.
   */
  #[DataProvider('tableNameProvider')]
  public function testGetFtsTableName(string $index_id, string $expected): void {
    $this->assertSame($expected, $this->schemaManager->getFtsTableName($index_id));
  }

  /**
   * Tests getFieldDataTableName() generates correct table name.
   */
  #[DataProvider('fieldDataTableNameProvider')]
  public function testGetFieldDataTableName(string $index_id, string $expected): void {
    $this->assertSame($expected, $this->schemaManager->getFieldDataTableName($index_id));
  }

  /**
   * Tests getItemsTableName() generates correct table name.
   */
  #[DataProvider('itemsTableNameProvider')]
  public function testGetItemsTableName(string $index_id, string $expected): void {
    $this->assertSame($expected, $this->schemaManager->getItemsTableName($index_id));
  }

  /**
   * Data provider for FTS table name tests.
   *
   * @return array<array{string, string}>
   *   Test cases.
   */
  public static function tableNameProvider(): array {
    return [
      'simple index name' => ['my_index', 'my_index_fts'],
      'index with numbers' => ['index_123', 'index_123_fts'],
      'default index' => ['default', 'default_fts'],
      'long index name' => ['very_long_index_name', 'very_long_index_name_fts'],
    ];
  }

  /**
   * Data provider for field data table name tests.
   *
   * @return array<array{string, string}>
   *   Test cases.
   */
  public static function fieldDataTableNameProvider(): array {
    return [
      'simple index name' => ['my_index', 'my_index_field_data'],
      'index with numbers' => ['index_123', 'index_123_field_data'],
    ];
  }

  /**
   * Data provider for items table name tests.
   *
   * @return array<array{string, string}>
   *   Test cases.
   */
  public static function itemsTableNameProvider(): array {
    return [
      'simple index name' => ['my_index', 'my_index_items'],
      'index with numbers' => ['index_123', 'index_123_items'],
    ];
  }

  /**
   * Tests tablesExist() returns TRUE when all tables exist.
   */
  public function testTablesExistReturnsTrue(): void {
    $schema = $this->createMock(Schema::class);
    $schema->method('tableExists')
      ->willReturn(TRUE);

    $connection = $this->createMock(Connection::class);
    $connection->method('schema')->willReturn($schema);

    $this->connectionManager->method('getConnection')
      ->willReturn($connection);

    $this->assertTrue($this->schemaManager->tablesExist('my_index'));
  }

  /**
   * Tests tablesExist() returns FALSE when any table does not exist.
   */
  public function testTablesExistReturnsFalse(): void {
    $schema = $this->createMock(Schema::class);
    // First table exists, second doesn't.
    $schema->method('tableExists')
      ->willReturnOnConsecutiveCalls(TRUE, FALSE);

    $connection = $this->createMock(Connection::class);
    $connection->method('schema')->willReturn($schema);

    $this->connectionManager->method('getConnection')
      ->willReturn($connection);

    $this->assertFalse($this->schemaManager->tablesExist('missing_index'));
  }

  /**
   * Tests that table names are generated consistently.
   */
  public function testTableNamingConsistency(): void {
    $index_id = 'products';

    $fts = $this->schemaManager->getFtsTableName($index_id);
    $field_data = $this->schemaManager->getFieldDataTableName($index_id);
    $items = $this->schemaManager->getItemsTableName($index_id);

    // All should start with the index ID.
    $this->assertStringStartsWith($index_id . '_', $fts);
    $this->assertStringStartsWith($index_id . '_', $field_data);
    $this->assertStringStartsWith($index_id . '_', $items);

    // Each should have distinct suffix.
    $this->assertNotSame($fts, $field_data);
    $this->assertNotSame($fts, $items);
    $this->assertNotSame($field_data, $items);
  }

  /**
   * Tests getFts5ColumnBoosts() returns boost values in column order.
   */
  public function testGetFts5ColumnBoosts(): void {
    // Create mock fields with different boosts.
    $title_field = $this->createMock(FieldInterface::class);
    $title_field->method('getType')->willReturn('text');
    $title_field->method('getBoost')->willReturn(5.0);

    $body_field = $this->createMock(FieldInterface::class);
    $body_field->method('getType')->willReturn('text');
    $body_field->method('getBoost')->willReturn(1.0);

    $summary_field = $this->createMock(FieldInterface::class);
    $summary_field->method('getType')->willReturn('text');
    $summary_field->method('getBoost')->willReturn(2.5);

    // Non-fulltext field should be excluded.
    $status_field = $this->createMock(FieldInterface::class);
    $status_field->method('getType')->willReturn('boolean');
    $status_field->method('getBoost')->willReturn(1.0);

    $index = $this->createMock(IndexInterface::class);
    $index->method('getFields')->willReturn([
      'title' => $title_field,
      'body' => $body_field,
      'summary' => $summary_field,
      'status' => $status_field,
    ]);

    // Configure field type mapper - only text fields are fulltext.
    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturnCallback(fn(string $type): bool => $type === 'text');

    $boosts = $this->schemaManager->getFts5ColumnBoosts($index);

    // Should return boosts only for fulltext fields, in order.
    $this->assertCount(3, $boosts);
    $this->assertSame([5.0, 1.0, 2.5], $boosts);
  }

  /**
   * Tests getFts5ColumnBoosts() returns empty array for index with no text.
   */
  public function testGetFts5ColumnBoostsEmptyForNoFulltext(): void {
    $status_field = $this->createMock(FieldInterface::class);
    $status_field->method('getType')->willReturn('boolean');

    $index = $this->createMock(IndexInterface::class);
    $index->method('getFields')->willReturn([
      'status' => $status_field,
    ]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturn(FALSE);

    $boosts = $this->schemaManager->getFts5ColumnBoosts($index);

    $this->assertSame([], $boosts);
  }

}
