<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Search;

use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Search\ConditionHelperInterface;
use Drupal\search_api_sqlite\Search\QueryBuilder;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Unit tests for QueryBuilder pure methods.
 *
 * Tests methods that don't require database interaction:
 * - escapeTerm()
 * - parseSearchInput()
 * - buildMatchQuery()
 */
#[CoversClass(QueryBuilder::class)]
#[Group('search_api_sqlite')]
final class QueryBuilderTest extends UnitTestCase {

  /**
   * The query builder under test.
   */
  private QueryBuilder $queryBuilder;

  /**
   * The mocked connection manager.
   */
  private ConnectionManagerInterface&MockObject $connectionManager;

  /**
   * The mocked schema manager.
   */
  private SchemaManagerInterface&MockObject $schemaManager;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $conditionHelper = $this->createMock(ConditionHelperInterface::class);

    $this->queryBuilder = new QueryBuilder(
      $this->connectionManager,
      $this->schemaManager,
      $conditionHelper,
    );
  }

  /**
   * Tests escapeTerm with normal terms.
   */
  public function testEscapeTermNormal(): void {
    $this->assertEquals('hello', $this->queryBuilder->escapeTerm('hello'));
    $this->assertEquals('drupal', $this->queryBuilder->escapeTerm('drupal'));
    // Note: 'world' contains 'OR' (case-insensitive), so it gets quoted.
    $this->assertEquals('"world"', $this->queryBuilder->escapeTerm('world'));
  }

  /**
   * Tests escapeTerm with special characters.
   */
  public function testEscapeTermSpecialChars(): void {
    // Terms with special chars should be quoted.
    $result = $this->queryBuilder->escapeTerm('hello-world');
    $this->assertStringContainsString('"', $result);

    $result = $this->queryBuilder->escapeTerm('test:value');
    $this->assertStringContainsString('"', $result);
  }

  /**
   * Tests escapeTerm with quotes.
   */
  public function testEscapeTermWithQuotes(): void {
    // Embedded quotes should be escaped.
    $result = $this->queryBuilder->escapeTerm('say "hello"');
    $this->assertStringContainsString('""', $result);
  }

  /**
   * Tests parseSearchInput with simple terms.
   */
  public function testParseSearchInputSimpleTerms(): void {
    $result = $this->queryBuilder->parseSearchInput('hello world');

    $this->assertContains('hello', $result['terms']);
    $this->assertContains('world', $result['terms']);
    $this->assertEmpty($result['phrases']);
  }

  /**
   * Tests parseSearchInput with quoted phrases.
   */
  public function testParseSearchInputWithPhrases(): void {
    $result = $this->queryBuilder->parseSearchInput('find "exact phrase" here');

    $this->assertContains('exact phrase', $result['phrases']);
    $this->assertContains('find', $result['terms']);
    $this->assertContains('here', $result['terms']);
  }

  /**
   * Tests parseSearchInput filters short terms based on min_chars.
   */
  public function testParseSearchInputFiltersShortTerms(): void {
    // With min_chars=2, single-character terms should be filtered.
    $result = $this->queryBuilder->parseSearchInput('a I do it', 2);

    $this->assertNotContains('a', $result['terms']);
    $this->assertNotContains('I', $result['terms']);
    $this->assertContains('do', $result['terms']);
    $this->assertContains('it', $result['terms']);
  }

  /**
   * Tests parseSearchInput with only phrases.
   */
  public function testParseSearchInputOnlyPhrases(): void {
    $result = $this->queryBuilder->parseSearchInput('"first phrase" "second phrase"');

    $this->assertContains('first phrase', $result['phrases']);
    $this->assertContains('second phrase', $result['phrases']);
    $this->assertEmpty($result['terms']);
  }

  /**
   * Tests parseSearchInput with empty input.
   */
  public function testParseSearchInputEmpty(): void {
    $result = $this->queryBuilder->parseSearchInput('');

    $this->assertEmpty($result['terms']);
    $this->assertEmpty($result['phrases']);
  }

  /**
   * Tests buildMatchQuery basic functionality.
   */
  public function testBuildMatchQueryBasic(): void {
    $query = $this->createMockQuery('hello world');

    $matchQuery = $this->queryBuilder->buildMatchQuery(
      $query,
      ['title' => 'title', 'body' => 'body'],
    );

    $this->assertNotNull($matchQuery);
    $this->assertStringContainsString('hello', $matchQuery);
    $this->assertStringContainsString('world', $matchQuery);
  }

  /**
   * Tests buildMatchQuery with specific fields.
   */
  public function testBuildMatchQueryWithSpecificFields(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn('test');
    $query->method('getFulltextFields')->willReturn(['title']);

    $matchQuery = $this->queryBuilder->buildMatchQuery(
      $query,
      ['title' => 'title', 'body' => 'body'],
    );

    $this->assertNotNull($matchQuery);
    // Should only search in title column.
    $this->assertStringContainsString('{title}', $matchQuery);
  }

  /**
   * Tests buildMatchQuery with phrase mode.
   *
   * Note: Phrase mode doesn't automatically wrap the entire input in quotes.
   * Users must explicitly use quotes for phrase matching (e.g., "hello world").
   */
  public function testBuildMatchQueryPhraseMode(): void {
    // Use quoted phrase input for actual phrase matching.
    $query = $this->createMockQuery('"hello world"');

    $matchQuery = $this->queryBuilder->buildMatchQuery(
      $query,
      ['title' => 'title', 'body' => 'body'],
      MatchingMode::Phrase,
    );

    $this->assertNotNull($matchQuery);
    // The phrase should be in the output.
    $this->assertStringContainsString('hello world', $matchQuery);
  }

  /**
   * Tests buildMatchQuery with prefix mode.
   */
  public function testBuildMatchQueryPrefixMode(): void {
    $query = $this->createMockQuery('drupal');

    $matchQuery = $this->queryBuilder->buildMatchQuery(
      $query,
      ['title' => 'title', 'body' => 'body'],
      MatchingMode::Prefix,
    );

    $this->assertNotNull($matchQuery);
    // Prefix mode should add asterisk.
    $this->assertStringContainsString('*', $matchQuery);
  }

  /**
   * Tests buildMatchQuery returns null for no keys.
   */
  public function testBuildMatchQueryNoKeys(): void {
    $query = $this->createMockQuery(NULL);

    $matchQuery = $this->queryBuilder->buildMatchQuery(
      $query,
      ['title' => 'title', 'body' => 'body'],
    );

    $this->assertNull($matchQuery);
  }

  /**
   * Tests buildMatchQuery with array keys.
   */
  public function testBuildMatchQueryArrayKeys(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn(['hello', 'world']);
    $query->method('getFulltextFields')->willReturn([]);

    $matchQuery = $this->queryBuilder->buildMatchQuery(
      $query,
      ['title' => 'title', 'body' => 'body'],
    );

    $this->assertNotNull($matchQuery);
    $this->assertStringContainsString('hello', $matchQuery);
    $this->assertStringContainsString('world', $matchQuery);
  }

  /**
   * Creates a mock Search API query for testing.
   *
   * @param string|array<mixed>|null $keys
   *   The search keys.
   *
   * @return \Drupal\search_api\Query\QueryInterface&\PHPUnit\Framework\MockObject\MockObject
   *   The mock query.
   */
  private function createMockQuery(string|array|null $keys): QueryInterface&MockObject {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn($keys);
    $query->method('getFulltextFields')->willReturn([]);

    return $query;
  }

}
