<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Enum;

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * FTS5 tokenizer types.
 *
 * Defines the available tokenizer options for SQLite FTS5 full-text search.
 */
enum Tokenizer: string {

  // Unicode61 tokenizer - handles Unicode text with diacritics removal.
  case Unicode61 = 'unicode61';

  // Porter stemmer tokenizer - reduces words to their root form.
  case Porter = 'porter';

  // ASCII tokenizer - simple tokenizer for basic ASCII text.
  case Ascii = 'ascii';

  // Trigram tokenizer - enables substring matching.
  case Trigram = 'trigram';

  /**
   * Gets a human-readable label for the tokenizer.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The tokenizer label.
   */
  public function label(): TranslatableMarkup {
    return match ($this) {
      self::Unicode61 => new TranslatableMarkup('Unicode61 (recommended)'),
      self::Porter => new TranslatableMarkup('Porter Stemmer'),
      self::Ascii => new TranslatableMarkup('ASCII'),
      self::Trigram => new TranslatableMarkup('Trigram (substring matching)'),
    };
  }

  /**
   * Gets a human-readable label for form options.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The form option label.
   */
  public function formLabel(): TranslatableMarkup {
    return match ($this) {
      self::Unicode61 => new TranslatableMarkup('Unicode61 (recommended for multilingual)'),
      self::Porter => new TranslatableMarkup('Porter Stemmer (English with word stemming)'),
      self::Ascii => new TranslatableMarkup('ASCII (basic English only)'),
      self::Trigram => new TranslatableMarkup('Trigram (substring matching for codes/SKUs)'),
    };
  }

  /**
   * Gets all tokenizers as form options.
   *
   * @return array<string, \Drupal\Core\StringTranslation\TranslatableMarkup>
   *   Array of tokenizer values to labels.
   */
  public static function formOptions(): array {
    $options = [];
    foreach (self::cases() as $case) {
      $options[$case->value] = $case->formLabel();
    }

    return $options;
  }

  /**
   * Gets all tokenizers as display labels.
   *
   * @return array<string, \Drupal\Core\StringTranslation\TranslatableMarkup>
   *   Array of tokenizer values to labels.
   */
  public static function labels(): array {
    $labels = [];
    foreach (self::cases() as $case) {
      $labels[$case->value] = $case->label();
    }

    return $labels;
  }

  /**
   * Checks if this tokenizer supports case sensitivity option.
   *
   * @return bool
   *   TRUE if the tokenizer supports case sensitivity.
   */
  public function supportsCaseSensitivity(): bool {
    return $this === self::Trigram;
  }

}
