<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use PHPUnit\Framework\Attributes\AfterClass;
use PHPUnit\Framework\Attributes\BeforeClass;

/**
 * Provides cleanup for SQLite database files in tests.
 *
 * Use this trait in kernel tests to ensure SQLite database files are properly
 * cleaned up between test runs. Cleanup runs automatically via PHPUnit
 * attributes - just add `use SqliteDatabaseCleanupTrait;` to your test class.
 */
trait SqliteDatabaseCleanupTrait {

  /**
   * Cleans up SQLite databases before any test runs.
   *
   * @beforeClass
   */
  #[BeforeClass]
  public static function cleanupSqliteDatabasesBeforeClass(): void {
    self::cleanupSqliteDatabases();
  }

  /**
   * Cleans up SQLite databases after all tests complete.
   *
   * @afterClass
   */
  #[AfterClass]
  public static function cleanupSqliteDatabasesAfterClass(): void {
    self::cleanupSqliteDatabases();
  }

  /**
   * Cleans up SQLite database files from test directories.
   */
  private static function cleanupSqliteDatabases(): void {
    // Clean sys_get_temp_dir() - works without Drupal.
    self::cleanupDirectory(sys_get_temp_dir() . '/search_api_sqlite_test');

    // On CI, various temp locations might be used.
    $possible_paths = [
      '/tmp/search_api_sqlite_test',
      sys_get_temp_dir() . '/simpletest/search_api_sqlite_test',
    ];

    foreach ($possible_paths as $path) {
      self::cleanupDirectory($path);
    }
  }

  /**
   * Removes all SQLite files from a directory.
   *
   * @param string $directory
   *   The directory path to clean.
   */
  private static function cleanupDirectory(string $directory): void {
    if (!is_dir($directory)) {
      return;
    }

    $files = glob($directory . '/*.sqlite*') ?: [];
    foreach ($files as $file) {
      @unlink($file);
    }
  }

}
