<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Search;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\search_api_sqlite\Search\ConditionHelper;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the ConditionHelper class.
 */
#[CoversClass(ConditionHelper::class)]
#[Group('search_api_sqlite')]
class ConditionHelperTest extends UnitTestCase {

  /**
   * The schema manager mock.
   */
  private MockObject&SchemaManagerInterface $schemaManager;

  /**
   * The field type mapper mock.
   */
  private MockObject&FieldTypeMapperInterface $fieldTypeMapper;

  /**
   * The condition helper under test.
   */
  private ConditionHelper $conditionHelper;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->fieldTypeMapper = $this->createMock(FieldTypeMapperInterface::class);

    $this->conditionHelper = new ConditionHelper(
      $this->schemaManager,
      $this->fieldTypeMapper,
    );
  }

  /**
   * Tests getValueColumn with different value types.
   */
  public function testGetValueColumn(): void {
    // Integer value.
    $this->assertEquals('value_integer', $this->conditionHelper->getValueColumn(42));

    // Boolean value.
    $this->assertEquals('value_integer', $this->conditionHelper->getValueColumn(TRUE));
    $this->assertEquals('value_integer', $this->conditionHelper->getValueColumn(FALSE));

    // Float value.
    $this->assertEquals('value_decimal', $this->conditionHelper->getValueColumn(3.14));

    // String value.
    $this->assertEquals('value_string', $this->conditionHelper->getValueColumn('text'));

    // Array - uses first element.
    $this->assertEquals('value_integer', $this->conditionHelper->getValueColumn([1, 2, 3]));
    $this->assertEquals('value_string', $this->conditionHelper->getValueColumn(['a', 'b']));
  }

  /**
   * Tests getValueColumnForField with field definition.
   */
  public function testGetValueColumnForFieldWithDefinition(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getType')->willReturn('integer');

    $fields = ['my_field' => $field];

    $this->fieldTypeMapper->expects($this->once())
      ->method('getStorageType')
      ->with('integer')
      ->willReturn('integer');

    $this->fieldTypeMapper->expects($this->once())
      ->method('getFieldDataColumn')
      ->with('integer')
      ->willReturn('value_integer');

    $result = $this->conditionHelper->getValueColumnForField('my_field', $fields, 42);
    $this->assertEquals('value_integer', $result);
  }

  /**
   * Tests getValueColumnForField fallback to value type detection.
   */
  public function testGetValueColumnForFieldFallback(): void {
    $fields = [];

    // No field definition - should fall back to value type.
    $result = $this->conditionHelper->getValueColumnForField('unknown_field', $fields, 'text');
    $this->assertEquals('value_string', $result);

    $result = $this->conditionHelper->getValueColumnForField('unknown_field', $fields, 123);
    $this->assertEquals('value_integer', $result);
  }

  /**
   * Tests buildLanguageSubquery with various operators.
   */
  public function testBuildLanguageSubquery(): void {
    $connection = $this->createMock(Connection::class);
    $subquery = $this->createMock(SelectInterface::class);

    $this->schemaManager->expects($this->once())
      ->method('getItemsTableName')
      ->with('test_index')
      ->willReturn('test_index_items');

    $connection->expects($this->once())
      ->method('select')
      ->with('test_index_items', 'it')
      ->willReturn($subquery);

    $subquery->expects($this->once())
      ->method('fields')
      ->with('it', ['item_id'])
      ->willReturnSelf();

    $subquery->expects($this->once())
      ->method('condition')
      ->with('it.language', 'en', 'IN')
      ->willReturnSelf();

    $result = $this->conditionHelper->buildLanguageSubquery(
      $connection,
      'test_index',
      'en',
      'IN'
    );

    $this->assertSame($subquery, $result);
  }

  /**
   * Tests buildLanguageSubquery with invalid operator.
   */
  public function testBuildLanguageSubqueryInvalidOperator(): void {
    $connection = $this->createMock(Connection::class);
    $subquery = $this->createMock(SelectInterface::class);

    $this->schemaManager->expects($this->once())
      ->method('getItemsTableName')
      ->with('test_index')
      ->willReturn('test_index_items');

    $connection->expects($this->once())
      ->method('select')
      ->with('test_index_items', 'it')
      ->willReturn($subquery);

    $subquery->expects($this->once())
      ->method('fields')
      ->with('it', ['item_id'])
      ->willReturnSelf();

    $result = $this->conditionHelper->buildLanguageSubquery(
      $connection,
      'test_index',
      'en',
      'INVALID'
    );

    $this->assertNull($result);
  }

  /**
   * Tests buildLanguageCondition with various operators.
   */
  public function testBuildLanguageCondition(): void {
    $connection = $this->createMock(Connection::class);
    $subquery = $this->createMock(SelectInterface::class);

    $this->schemaManager->expects($this->once())
      ->method('getItemsTableName')
      ->with('test_index')
      ->willReturn('test_index_items');

    $connection->expects($this->once())
      ->method('select')
      ->with('test_index_items', 'it')
      ->willReturn($subquery);

    $subquery->expects($this->once())
      ->method('fields')
      ->with('it', ['item_id'])
      ->willReturnSelf();

    $subquery->expects($this->once())
      ->method('condition')
      ->with('it.language', 'en')
      ->willReturnSelf();

    [$result_subquery, $outer_operator] = $this->conditionHelper->buildLanguageCondition(
      $connection,
      'test_index',
      'en',
      '='
    );

    $this->assertSame($subquery, $result_subquery);
    $this->assertEquals('IN', $outer_operator);
  }

  /**
   * Tests buildLanguageCondition with NOT IN operator.
   */
  public function testBuildLanguageConditionNotIn(): void {
    $connection = $this->createMock(Connection::class);
    $subquery = $this->createMock(SelectInterface::class);

    $this->schemaManager->method('getItemsTableName')->willReturn('test_index_items');
    $connection->method('select')->willReturn($subquery);
    $subquery->method('fields')->willReturnSelf();
    $subquery->method('condition')->willReturnSelf();

    [$result_subquery, $outer_operator] = $this->conditionHelper->buildLanguageCondition(
      $connection,
      'test_index',
      'en',
      'NOT IN'
    );

    $this->assertSame($subquery, $result_subquery);
    $this->assertEquals('NOT IN', $outer_operator);
  }

}
