<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Event;

use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\QueryInterface;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * Event fired before executing facet queries.
 *
 * Allows subscribers to alter facet configurations before facets are
 * calculated. This can be used for:
 * - Adding dynamic facets
 * - Modifying facet settings (min_count, limit, etc.)
 * - Removing facets conditionally
 * - Adjusting operator or missing settings.
 *
 * @see \Drupal\search_api_sqlite\Event\SearchApiSqliteEvents::FACETS_PRE_EXECUTE
 */
final class FacetsPreExecuteEvent extends Event {

  /**
   * Constructs a FacetsPreExecuteEvent.
   *
   * @param array<string, array<string, mixed>> $facets
   *   The facets configuration array, keyed by facet ID.
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The Search API query being executed.
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param int $resultCount
   *   The total number of search results before facet filtering.
   */
  public function __construct(
    private array $facets,
    private readonly QueryInterface $query,
    private readonly IndexInterface $index,
    private readonly int $resultCount,
  ) {}

  /**
   * Gets the facets configuration.
   *
   * @return array<string, array<string, mixed>>
   *   The facets configuration array, keyed by facet ID. Each facet contains:
   *   - field: The field name to facet on.
   *   - limit: Maximum number of facet values to return.
   *   - min_count: Minimum count for a value to be included.
   *   - missing: Whether to include a "missing" facet for items without value.
   *   - operator: 'and' or 'or' for combining with other facets.
   */
  public function getFacets(): array {
    return $this->facets;
  }

  /**
   * Sets the facets configuration.
   *
   * @param array<string, array<string, mixed>> $facets
   *   The new facets configuration array.
   *
   * @return $this
   */
  public function setFacets(array $facets): self {
    $this->facets = $facets;
    return $this;
  }

  /**
   * Adds or updates a facet.
   *
   * @param string $facetId
   *   The facet ID.
   * @param array<string, mixed> $facetConfig
   *   The facet configuration.
   *
   * @return $this
   */
  public function setFacet(string $facetId, array $facetConfig): self {
    $this->facets[$facetId] = $facetConfig;
    return $this;
  }

  /**
   * Removes a facet.
   *
   * @param string $facetId
   *   The facet ID to remove.
   *
   * @return $this
   */
  public function removeFacet(string $facetId): self {
    unset($this->facets[$facetId]);
    return $this;
  }

  /**
   * Gets a specific facet configuration.
   *
   * @param string $facetId
   *   The facet ID.
   *
   * @return array<string, mixed>|null
   *   The facet configuration or NULL if not found.
   */
  public function getFacet(string $facetId): ?array {
    return $this->facets[$facetId] ?? NULL;
  }

  /**
   * Gets the Search API query.
   *
   * @return \Drupal\search_api\Query\QueryInterface
   *   The query being executed.
   */
  public function getQuery(): QueryInterface {
    return $this->query;
  }

  /**
   * Gets the search index.
   *
   * @return \Drupal\search_api\IndexInterface
   *   The search index.
   */
  public function getIndex(): IndexInterface {
    return $this->index;
  }

  /**
   * Gets the total number of search results.
   *
   * @return int
   *   The result count before facet filtering.
   */
  public function getResultCount(): int {
    return $this->resultCount;
  }

}
