<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Event;

use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\QueryInterface;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * Event fired before building the FTS5 MATCH query.
 *
 * Allows subscribers to alter the MATCH query string before it is sent to
 * SQLite FTS5. This can be used for:
 * - Custom boosting of specific terms or fields
 * - Adding synonyms or query expansion
 * - Modifying FTS5 query syntax
 * - Adding proximity operators.
 *
 * @see \Drupal\search_api_sqlite\Event\SearchApiSqliteEvents::MATCH_QUERY_ALTER
 */
final class MatchQueryAlterEvent extends Event {

  /**
   * Constructs a MatchQueryAlterEvent.
   *
   * @param string $matchQuery
   *   The FTS5 MATCH query string.
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The Search API query being executed.
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param array<string> $fulltextFields
   *   The fulltext fields being searched.
   */
  public function __construct(
    private string $matchQuery,
    private readonly QueryInterface $query,
    private readonly IndexInterface $index,
    private readonly array $fulltextFields,
  ) {}

  /**
   * Gets the FTS5 MATCH query string.
   *
   * @return string
   *   The MATCH query string.
   */
  public function getMatchQuery(): string {
    return $this->matchQuery;
  }

  /**
   * Sets the FTS5 MATCH query string.
   *
   * @param string $matchQuery
   *   The new MATCH query string.
   *
   * @return $this
   */
  public function setMatchQuery(string $matchQuery): self {
    $this->matchQuery = $matchQuery;
    return $this;
  }

  /**
   * Gets the Search API query.
   *
   * @return \Drupal\search_api\Query\QueryInterface
   *   The query being executed.
   */
  public function getQuery(): QueryInterface {
    return $this->query;
  }

  /**
   * Gets the search index.
   *
   * @return \Drupal\search_api\IndexInterface
   *   The search index.
   */
  public function getIndex(): IndexInterface {
    return $this->index;
  }

  /**
   * Gets the fulltext fields being searched.
   *
   * @return array<string>
   *   Array of fulltext field names.
   */
  public function getFulltextFields(): array {
    return $this->fulltextFields;
  }

}
