<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\search_api\Query\ResultSetInterface;

/**
 * Defines the interface for FTS5 native highlighting.
 *
 * This service adds FTS5 snippet expressions to search queries and
 * processes the results to set excerpts on result items.
 */
interface HighlighterInterface {

  /**
   * Adds snippet expressions to a search query.
   *
   * Modifies the query to include FTS5 snippet() expressions for each
   * fulltext field, enabling native highlighting.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $db_query
   *   The database query to modify.
   * @param string $fts_table
   *   The FTS5 table name.
   * @param array<string, string> $fulltext_fields
   *   Map of field IDs to sanitized column names.
   * @param array{prefix?: string, suffix?: string, excerpt_length?: int, exclude_fields?: array<string>} $config
   *   Highlight configuration with keys:
   *   - prefix: HTML tag to wrap matches (default: '<mark>')
   *   - suffix: Closing tag (default: '</mark>')
   *   - excerpt_length: Approximate excerpt length in tokens (default: 64)
   *   - exclude_fields: Fields to exclude from highlighting.
   */
  public function addSnippetExpressions(
    SelectInterface $db_query,
    string $fts_table,
    array $fulltext_fields,
    array $config,
  ): void;

  /**
   * Processes search results to set excerpts from snippet data.
   *
   * Reads snippet columns from result rows and sets the excerpt on
   * corresponding result items.
   *
   * @param \Drupal\search_api\Query\ResultSetInterface $results
   *   The search results.
   * @param array<object> $rows
   *   The raw database result rows containing snippet columns.
   * @param array<string, string> $fulltext_fields
   *   Map of field IDs to sanitized column names.
   * @param array{prefix?: string, suffix?: string, excerpt_length?: int, exclude_fields?: array<string>} $config
   *   Highlight configuration.
   */
  public function processSnippetResults(
    ResultSetInterface $results,
    array $rows,
    array $fulltext_fields,
    array $config,
  ): void;

}
