<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Entity\Server;
use Drupal\Tests\user\Traits\UserCreationTrait;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use Drupal\search_api_sqlite\Hook\EntityHooks;
use Drupal\search_api_sqlite\Hook\IndexFormHooks;

/**
 * Tests that hooks are properly registered and functional.
 *
 * Tests both the procedural hook delegation (D10.x) and OOP hook classes.
 */
#[CoversClass(EntityHooks::class)]
#[CoversClass(IndexFormHooks::class)]
#[Group('search_api_sqlite')]
class HooksTest extends KernelTestBase {

  use SqliteDatabaseCleanupTrait;
  use UserCreationTrait;

  /**
   * {@inheritdoc}
   *
   * @var list<string>
   */
  protected static $modules = [
    'system',
    'user',
    'search_api',
    'search_api_sqlite',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('search_api_task');
    $this->installSchema('search_api', ['search_api_item']);
    $this->installConfig(['search_api']);

    // Ensure the module file is loaded.
    \Drupal::moduleHandler()->loadInclude('search_api_sqlite', 'module');
  }

  /**
   * Tests that hook_entity_operation is registered and callable.
   */
  public function testEntityOperationHookIsRegistered(): void {
    $module_handler = \Drupal::moduleHandler();

    // Check module implements the hook by invoking it.
    // This works on both D10.x and D11.x.
    $this->assertTrue(
      $module_handler->hasImplementations('entity_operation'),
      'hook_entity_operation has implementations.'
    );

    // Verify our module's hook function exists.
    $this->assertTrue(
      function_exists('search_api_sqlite_entity_operation'),
      'search_api_sqlite_entity_operation function exists.'
    );
  }

  /**
   * Tests that hook_form_search_api_index_form_alter is registered.
   */
  public function testFormAlterHookIsRegistered(): void {
    $module_handler = \Drupal::moduleHandler();

    // Check hook has implementations.
    $this->assertTrue(
      $module_handler->hasImplementations('form_search_api_index_form_alter'),
      'hook_form_search_api_index_form_alter has implementations.'
    );

    // Verify our module's hook function exists.
    $this->assertTrue(
      function_exists('search_api_sqlite_form_search_api_index_form_alter'),
      'search_api_sqlite_form_search_api_index_form_alter function exists.'
    );
  }

  /**
   * Tests hook_entity_operation adds SQLite Operations link for SQLite indexes.
   */
  public function testEntityOperationAddsLinkForSqliteIndex(): void {
    // Create a server with SQLite backend.
    $server = Server::create([
      'id' => 'test_sqlite_server',
      'name' => 'Test SQLite Server',
      'backend' => 'search_api_sqlite',
      'backend_config' => [
        'database_path' => 'temporary://search_api_sqlite_hooks_test',
      ],
    ]);
    $server->save();

    // Create an index using that server.
    $index = Index::create([
      'id' => 'test_sqlite_index',
      'name' => 'Test SQLite Index',
      'server' => 'test_sqlite_server',
      'datasource_settings' => [],
      'tracker_settings' => [
        'default' => [],
      ],
    ]);
    $index->save();

    // Set current user with admin permission.
    $this->setUpCurrentUser([], ['administer search_api']);

    // Invoke hook_entity_operation.
    $operations = \Drupal::moduleHandler()->invokeAll('entity_operation', [$index]);

    $this->assertArrayHasKey('sqlite_operations', $operations, 'SQLite Operations link is added.');
    $this->assertEquals('SQLite Operations', (string) $operations['sqlite_operations']['title']);
  }

  /**
   * Tests hook_entity_operation does not add link for non-SQLite indexes.
   */
  public function testEntityOperationNoLinkForNonSqliteIndex(): void {
    // For this test, we create an index without a server configured.
    $index = Index::create([
      'id' => 'test_no_server_index',
      'name' => 'Test Index Without Server',
      'datasource_settings' => [],
      'tracker_settings' => [
        'default' => [],
      ],
    ]);
    $index->save();

    // Set current user with admin permission.
    $this->setUpCurrentUser([], ['administer search_api']);

    // Invoke hook_entity_operation.
    $operations = \Drupal::moduleHandler()->invokeAll('entity_operation', [$index]);

    $this->assertArrayNotHasKey('sqlite_operations', $operations, 'SQLite Operations link is not added for non-SQLite index.');
  }

  /**
   * Tests hook_entity_operation respects permissions.
   */
  public function testEntityOperationRespectsPermissions(): void {
    // Create a server with SQLite backend.
    $server = Server::create([
      'id' => 'test_sqlite_server_perm',
      'name' => 'Test SQLite Server',
      'backend' => 'search_api_sqlite',
      'backend_config' => [
        'database_path' => 'temporary://search_api_sqlite_hooks_test',
      ],
    ]);
    $server->save();

    // Create an index using that server.
    $index = Index::create([
      'id' => 'test_sqlite_index_perm',
      'name' => 'Test SQLite Index',
      'server' => 'test_sqlite_server_perm',
      'datasource_settings' => [],
      'tracker_settings' => [
        'default' => [],
      ],
    ]);
    $index->save();

    // Set current user WITHOUT admin permission.
    $this->setUpCurrentUser([], []);

    // Invoke hook_entity_operation.
    $operations = \Drupal::moduleHandler()->invokeAll('entity_operation', [$index]);

    $this->assertArrayNotHasKey('sqlite_operations', $operations, 'SQLite Operations link is not shown without permission.');
  }

  /**
   * Tests hook_entity_operation ignores non-index entities.
   */
  public function testEntityOperationIgnoresNonIndexEntities(): void {
    // Create a server entity (not an index).
    $server = Server::create([
      'id' => 'test_server_entity',
      'name' => 'Test Server',
      'backend' => 'search_api_sqlite',
      'backend_config' => [
        'database_path' => 'temporary://search_api_sqlite_hooks_test',
      ],
    ]);
    $server->save();

    // Set current user with admin permission.
    $this->setUpCurrentUser([], ['administer search_api']);

    // Invoke hook_entity_operation with a server (not index).
    $operations = \Drupal::moduleHandler()->invokeAll('entity_operation', [$server]);

    $this->assertArrayNotHasKey('sqlite_operations', $operations, 'SQLite Operations link is not added for server entities.');
  }

}
