<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Utility\Utility;
use Drupal\search_api_sqlite\Plugin\search_api\backend\SqliteFts5;
use Drupal\search_api_sqlite\Plugin\search_api\processor\SpellCheck;
use Drupal\search_api_sqlite\Spellcheck\SpellCheckHandler;
use Drupal\Tests\search_api\Functional\ExampleContentTrait;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests spellcheck functionality of the SQLite FTS5 backend.
 */
#[CoversClass(SqliteFts5::class)]
#[CoversClass(SpellCheckHandler::class)]
#[CoversClass(SpellCheck::class)]
#[Group('search_api_sqlite')]
class SpellcheckTest extends KernelTestBase {

  use SqliteDatabaseCleanupTrait;
  use ExampleContentTrait;

  /**
   * {@inheritdoc}
   *
   * @var list<string>
   */
  protected static $modules = [
    'entity_test',
    'field',
    'system',
    'filter',
    'text',
    'user',
    'search_api',
    'search_api_sqlite',
    'search_api_test',
    'search_api_test_sqlite',
    'search_api_test_example_content',
    'search_api_sqlite_test_spellcheck',
  ];

  /**
   * The search index ID.
   */
  protected string $indexId = 'sapi_sqlite_spellcheck_index';

  /**
   * The spellcheck handler service.
   */
  protected SpellCheckHandler $spellCheckHandler;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->installSchema('search_api', ['search_api_item']);
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('entity_test_mulrev_changed');
    $this->installEntitySchema('search_api_task');
    $this->installConfig('search_api');

    // Do not use a batch for tracking the initial items after creating an
    // index when running the tests via the GUI.
    if (!Utility::isRunningInCli()) {
      \Drupal::state()->set('search_api_use_tracking_batch', FALSE);
    }

    $this->installConfig([
      'search_api_test_example_content',
      'search_api_test_sqlite',
      'search_api_sqlite_test_spellcheck',
    ]);

    $this->spellCheckHandler = $this->container->get('search_api_sqlite.spellcheck_handler');

    $this->setUpExampleStructure();
    $this->insertExampleContent();

    $this->indexItems($this->indexId);
  }

  /**
   * Tests that the spellcheck processor is enabled on the index.
   */
  public function testSpellcheckProcessorEnabled(): void {
    $index = Index::load($this->indexId);
    $this->assertNotNull($index, 'Index loaded successfully.');

    // Check processor is enabled.
    $this->assertTrue($index->isValidProcessor('search_api_sqlite_spellcheck'));

    // Get the processor and check configuration.
    $processor = $index->getProcessor('search_api_sqlite_spellcheck');
    $this->assertInstanceOf(SpellCheck::class, $processor);

    $config = $processor->getConfiguration();
    $this->assertEquals('aspell', $config['backend']);
    $this->assertEquals('fixed', $config['language_mode']);
    $this->assertEquals('en', $config['language']);
    $this->assertEquals(5, $config['result_threshold']);
    $this->assertEquals(3, $config['max_suggestions']);
  }

  /**
   * Tests spellcheck feature flag is declared.
   */
  public function testSpellcheckFeatureFlag(): void {
    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    $server = $index->getServerInstance();
    $this->assertNotNull($server);

    $backend = $server->getBackend();
    $this->assertInstanceOf(SqliteFts5::class, $backend);

    // Check that the backend declares spellcheck support.
    $features = $backend->getSupportedFeatures();
    $this->assertContains('search_api_spellcheck', $features);
  }

  /**
   * Tests that processor only supports SQLite backend indexes.
   */
  public function testProcessorSupportsOnlySqliteBackend(): void {
    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    // The processor should support this index (SQLite backend).
    $this->assertTrue(SpellCheck::supportsIndex($index));
  }

  /**
   * Tests spellcheck suggestions for misspelled query.
   */
  public function testSpellcheckSuggestionsForMisspelling(): void {
    $backends = $this->spellCheckHandler->getAvailableBackends();
    if ($backends === []) {
      $this->markTestSkipped('No spell check backends available (aspell/hunspell/pspell not installed).');
    }

    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    // Test the handler directly first.
    // Use "tset" which is a common misspelling of "test".
    $result = $this->spellCheckHandler->checkQuery('tset', 'aspell', 'en');
    $this->assertTrue($result['has_misspellings'], 'Handler should detect misspelling in "tset".');
    $this->assertNotEmpty($result['collation'], 'Handler should return collation.');
    $this->assertNotEmpty($result['suggestions'], 'Handler should return per-word suggestions.');
    $this->assertArrayHasKey('tset', $result['suggestions'], 'Suggestions keyed by word.');

    // Search for a misspelled word that should have 0 results.
    // "tset" is misspelled version of "test" which exists in the test content.
    $query = $index->query();
    $query->keys('tset');
    $query->setOption('search_api_spellcheck', TRUE);

    $results = $query->execute();

    // With no matching content, results should be empty or below threshold.
    $result_count = $results->getResultCount();
    $this->assertLessThan(5, $result_count, 'Results below threshold for misspelled query.');

    // Check if spellcheck extra data was populated with suggestions.
    // The suggestion "test" should be validated against the FTS5 index
    // and returned because "test" exists in our test content.
    $spellcheck = $results->getExtraData('search_api_spellcheck');
    $this->assertIsArray($spellcheck, 'Spellcheck extra data should be an array.');
    $this->assertArrayHasKey('collation', $spellcheck, 'Spellcheck should have collation.');
    $this->assertNotEmpty($spellcheck['collation'], 'Spellcheck collation should not be empty.');
    $this->assertStringContainsString('test', $spellcheck['collation'], 'Collation contains corrected word.');
    // Also check per-word suggestions for SuggestionsSpellCheck plugin.
    $this->assertArrayHasKey('suggestions', $spellcheck, 'Spellcheck should have suggestions.');
    $this->assertArrayHasKey('tset', $spellcheck['suggestions'], 'Suggestions keyed by misspelled word.');
  }

  /**
   * Tests no spellcheck suggestions when results are above threshold.
   */
  public function testNoSpellcheckWhenResultsAboveThreshold(): void {
    $backends = $this->spellCheckHandler->getAvailableBackends();
    if ($backends === []) {
      $this->markTestSkipped('No spell check backends available.');
    }

    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    // Search for "test" which should match multiple items.
    $query = $index->query();
    $query->keys('test');
    $query->setOption('search_api_spellcheck', TRUE);

    $results = $query->execute();

    // Results should be above threshold (>= 5).
    $result_count = $results->getResultCount();

    // Check spellcheck extra data - no suggestions if above threshold.
    $spellcheck = $results->getExtraData('search_api_spellcheck');
    if ($result_count >= 5) {
      // If we have enough results, spellcheck should be skipped.
      $this->assertTrue(
        $spellcheck === NULL || empty($spellcheck['collation']),
        'Spellcheck skipped when results above threshold.'
      );
    }
  }

  /**
   * Tests spellcheck is skipped when processor is disabled.
   */
  public function testSpellcheckSkippedWhenProcessorDisabled(): void {
    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    // Disable the spellcheck processor.
    $index->set('processor_settings', array_filter(
      $index->get('processor_settings'),
      fn($key): bool => $key !== 'search_api_sqlite_spellcheck',
      ARRAY_FILTER_USE_KEY,
    ));
    $index->save();

    // Verify processor is disabled.
    $this->assertFalse($index->isValidProcessor('search_api_sqlite_spellcheck'));

    // Re-index after config change.
    $this->indexItems($this->indexId);

    // Search should not populate spellcheck suggestions.
    $query = $index->query();
    $query->keys('mispeled');
    $query->setOption('search_api_spellcheck', TRUE);

    // Execute the query.
    $query->execute();

    // Spellcheck option should remain as TRUE (not populated with array).
    $spellcheck = $query->getOption('search_api_spellcheck');
    $this->assertTrue($spellcheck === TRUE, 'Spellcheck option unchanged when processor disabled.');
  }

  /**
   * Tests processor default configuration.
   */
  public function testProcessorDefaultConfiguration(): void {
    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    // Create a new processor instance to check defaults.
    /** @var \Drupal\search_api\Processor\ProcessorPluginManager $manager */
    $manager = \Drupal::service('plugin.manager.search_api.processor');
    /** @var \Drupal\search_api_sqlite\Plugin\search_api\processor\SpellCheck $processor */
    $processor = $manager->createInstance('search_api_sqlite_spellcheck', [
      '#index' => $index,
    ]);

    $defaults = $processor->defaultConfiguration();
    $this->assertEquals('aspell', $defaults['backend']);
    $this->assertEquals('auto', $defaults['language_mode']);
    $this->assertEquals('en', $defaults['language']);
    $this->assertEquals(5, $defaults['result_threshold']);
    $this->assertEquals(3, $defaults['max_suggestions']);
  }

  /**
   * Tests that suggestions not in the index are filtered out.
   */
  public function testSuggestionsFilteredByIndexContent(): void {
    $backends = $this->spellCheckHandler->getAvailableBackends();
    if ($backends === []) {
      $this->markTestSkipped('No spell check backends available.');
    }

    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    // Search for a misspelled word where the correction doesn't exist in index.
    // "articel" → "article" but "article" is not in the test content.
    $query = $index->query();
    $query->keys('articel');
    $query->setOption('search_api_spellcheck', TRUE);

    // Execute the query - results not needed, we check the spellcheck option.
    $query->execute();

    // The spellcheck should remain TRUE (not populated) because the
    // suggestion "article" doesn't exist in the index content.
    $spellcheck = $query->getOption('search_api_spellcheck');
    $this->assertTrue(
      $spellcheck === TRUE,
      'Spellcheck should remain TRUE when suggestion does not exist in index.',
    );
  }

}
