<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Database;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\File\FileSystemInterface;
use Drupal\search_api_sqlite\Database\ConnectionManager;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the ConnectionManager service.
 */
#[CoversClass(ConnectionManager::class)]
#[Group('search_api_sqlite')]
final class ConnectionManagerTest extends UnitTestCase {

  /**
   * The mocked file system service.
   */
  private FileSystemInterface&MockObject $fileSystem;

  /**
   * The mocked config factory.
   */
  private ConfigFactoryInterface&MockObject $configFactory;

  /**
   * The mocked logger.
   */
  private LoggerInterface&MockObject $logger;

  /**
   * The mocked config object.
   */
  private ImmutableConfig&MockObject $config;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->fileSystem = $this->createMock(FileSystemInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);
    $this->config = $this->createMock(ImmutableConfig::class);

    $this->configFactory->method('get')
      ->with('search_api_sqlite.settings')
      ->willReturn($this->config);
  }

  /**
   * Creates a ConnectionManager instance for testing.
   *
   * @return \Drupal\search_api_sqlite\Database\ConnectionManager
   *   The connection manager.
   */
  private function createConnectionManager(): ConnectionManager {
    return new ConnectionManager(
      $this->fileSystem,
      $this->configFactory,
      $this->logger
    );
  }

  /**
   * Tests getDatabasePath() returns correct path with default config.
   */
  public function testGetDatabasePathWithDefaultConfig(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn(NULL);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        ['private://search_api_sqlite', FALSE],
        ['private://', '/var/private'],
      ]);

    $connectionManager = $this->createConnectionManager();
    $path = $connectionManager->getDatabasePath('test_index');

    $this->assertSame('/var/private/search_api_sqlite/test_index.sqlite', $path);
  }

  /**
   * Tests getDatabasePath() returns correct path with custom config.
   */
  public function testGetDatabasePathWithCustomConfig(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn('/custom/path');

    $this->fileSystem->method('realpath')
      ->with('/custom/path')
      ->willReturn('/custom/path');

    $connectionManager = $this->createConnectionManager();
    $path = $connectionManager->getDatabasePath('my_index');

    $this->assertSame('/custom/path/my_index.sqlite', $path);
  }

  /**
   * Tests getDatabasePath() throws exception when private path not configured.
   */
  public function testGetDatabasePathThrowsWhenPrivateNotConfigured(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn(NULL);

    $this->fileSystem->method('realpath')
      ->willReturn(FALSE);

    $connectionManager = $this->createConnectionManager();

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Private file system is not configured');

    $connectionManager->getDatabasePath('test_index');
  }

  /**
   * Tests databaseExists() returns true when file exists.
   */
  public function testDatabaseExistsReturnsTrueWhenFileExists(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn(NULL);

    // Use a real temporary directory for this test.
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);
    $dbPath = $tempDir . '/test.sqlite';
    touch($dbPath);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        ['private://search_api_sqlite', $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();
    $exists = $connectionManager->databaseExists('test');

    $this->assertTrue($exists);

    // Cleanup.
    unlink($dbPath);
    rmdir($tempDir);
  }

  /**
   * Tests databaseExists() returns false when file does not exist.
   */
  public function testDatabaseExistsReturnsFalseWhenFileNotExists(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn('/nonexistent/path');

    $this->fileSystem->method('realpath')
      ->with('/nonexistent/path')
      ->willReturn('/nonexistent/path');

    $connectionManager = $this->createConnectionManager();
    $exists = $connectionManager->databaseExists('test_index');

    $this->assertFalse($exists);
  }

  /**
   * Tests deleteDatabase() returns true and cleans up files.
   */
  public function testDeleteDatabaseRemovesAllFiles(): void {
    // Create temporary test files.
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);
    $dbPath = $tempDir . '/myindex.sqlite';
    $walPath = $dbPath . '-wal';
    $shmPath = $dbPath . '-shm';
    touch($dbPath);
    touch($walPath);
    touch($shmPath);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->with($tempDir)
      ->willReturn($tempDir);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        'Deleted SQLite database for index @index',
        ['@index' => 'myindex']
      );

    $connectionManager = $this->createConnectionManager();
    $result = $connectionManager->deleteDatabase('myindex');

    $this->assertTrue($result);
    $this->assertFileDoesNotExist($dbPath);
    $this->assertFileDoesNotExist($walPath);
    $this->assertFileDoesNotExist($shmPath);

    // Cleanup.
    rmdir($tempDir);
  }

  /**
   * Tests deleteDatabase() returns true when no files exist.
   */
  public function testDeleteDatabaseReturnsTrueWhenNoFiles(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn('/tmp/nonexistent');

    $this->fileSystem->method('realpath')
      ->with('/tmp/nonexistent')
      ->willReturn('/tmp/nonexistent');

    // Logger should still log the deletion.
    $this->logger->expects($this->once())
      ->method('info');

    $connectionManager = $this->createConnectionManager();
    $result = $connectionManager->deleteDatabase('nonexistent_index');

    $this->assertTrue($result);
  }

  /**
   * Tests closeConnection() removes cached connection.
   */
  public function testCloseConnectionRemovesCachedConnection(): void {
    // Create a temp directory and database.
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        [$tempDir, $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();

    // Get connection to cache it.
    $connection = $connectionManager->getConnection('cache_test');

    // Close should clear the cache (no exception = success).
    $connectionManager->closeConnection('cache_test');

    // Verify we can get a fresh connection after closing.
    $newConnection = $connectionManager->getConnection('cache_test');
    $this->assertNotSame($connection, $newConnection);

    // Close should clear the cache.
    $connectionManager->closeConnection('cache_test');

    // Cleanup.
    $connectionManager->deleteDatabase('cache_test');
    rmdir($tempDir);
  }

  /**
   * Tests shouldUseTempTable returns false below threshold.
   */
  public function testShouldUseTempTableReturnsFalseBelowThreshold(): void {
    $connectionManager = $this->createConnectionManager();

    $this->assertFalse($connectionManager->shouldUseTempTable(0));
    $this->assertFalse($connectionManager->shouldUseTempTable(100));
    $this->assertFalse($connectionManager->shouldUseTempTable(499));
  }

  /**
   * Tests shouldUseTempTable returns true at or above threshold.
   */
  public function testShouldUseTempTableReturnsTrueAtThreshold(): void {
    $connectionManager = $this->createConnectionManager();

    $this->assertTrue($connectionManager->shouldUseTempTable(500));
    $this->assertTrue($connectionManager->shouldUseTempTable(1000));
    $this->assertTrue($connectionManager->shouldUseTempTable(10000));
  }

  /**
   * Tests getTempTableThreshold returns expected value.
   */
  public function testGetTempTableThresholdReturnsValue(): void {
    $connectionManager = $this->createConnectionManager();

    $this->assertSame(500, $connectionManager->getTempTableThreshold());
  }

  /**
   * Tests createTempItemsTable creates a table with item IDs.
   */
  public function testCreateTempItemsTableCreatesTableWithItems(): void {
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        [$tempDir, $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();

    $itemIds = ['item_1', 'item_2', 'item_3'];
    $tableName = $connectionManager->createTempItemsTable('test_idx', $itemIds, 'test');

    // Verify table name format.
    $this->assertStringStartsWith('temp_test_', $tableName);

    // Verify items were inserted by querying the temp table.
    $connection = $connectionManager->getConnection('test_idx');
    $result = $connection->select($tableName, 't')
      ->fields('t', ['item_id'])
      ->orderBy('item_id')
      ->execute();
    $results = $result !== NULL ? $result->fetchCol() : [];

    $this->assertSame(['item_1', 'item_2', 'item_3'], $results);

    // Cleanup.
    $connectionManager->dropTempTable('test_idx', $tableName);
    $connectionManager->deleteDatabase('test_idx');
    rmdir($tempDir);
  }

  /**
   * Tests createTempItemsTable batches large item sets.
   */
  public function testCreateTempItemsTableBatchesLargeItemSets(): void {
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        [$tempDir, $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();

    // Create 1200 items (should be batched: 500 + 500 + 200).
    $itemIds = [];
    for ($i = 1; $i <= 1200; $i++) {
      $itemIds[] = 'item_' . $i;
    }

    $tableName = $connectionManager->createTempItemsTable('test_idx', $itemIds, 'batch');

    // Verify all items were inserted.
    $connection = $connectionManager->getConnection('test_idx');
    $result = $connection->select($tableName, 't')
      ->countQuery()
      ->execute();
    $count = $result !== NULL ? (int) $result->fetchField() : 0;

    $this->assertSame(1200, $count);

    // Cleanup.
    $connectionManager->dropTempTable('test_idx', $tableName);
    $connectionManager->deleteDatabase('test_idx');
    rmdir($tempDir);
  }

  /**
   * Tests dropTempTable removes the table.
   */
  public function testDropTempTableRemovesTable(): void {
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        [$tempDir, $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();

    $itemIds = ['item_1', 'item_2'];
    $tableName = $connectionManager->createTempItemsTable('test_idx', $itemIds, 'drop');

    // Verify table exists.
    $connection = $connectionManager->getConnection('test_idx');
    $result = $connection->select($tableName, 't')
      ->countQuery()
      ->execute();
    $count = $result !== NULL ? (int) $result->fetchField() : 0;
    $this->assertSame(2, $count);

    // Drop the table.
    $connectionManager->dropTempTable('test_idx', $tableName);

    // Verify table is gone (query should fail).
    $this->expectException(\Exception::class);
    $connection->select($tableName, 't')->countQuery()->execute();
  }

  /**
   * Tests createTempItemsTable clears existing data on reuse.
   */
  public function testCreateTempItemsTableClearsExistingData(): void {
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        [$tempDir, $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();

    // Create first set of items.
    $tableName1 = $connectionManager->createTempItemsTable(
      'test_idx',
      ['item_a', 'item_b', 'item_c'],
      'reuse'
    );

    // Create second set with same purpose (should reuse table name).
    $tableName2 = $connectionManager->createTempItemsTable(
      'test_idx',
      ['item_x', 'item_y'],
      'reuse'
    );

    // Table names should be the same.
    $this->assertSame($tableName1, $tableName2);

    // Only the second set should be in the table.
    $connection = $connectionManager->getConnection('test_idx');
    $result = $connection->select($tableName2, 't')
      ->fields('t', ['item_id'])
      ->orderBy('item_id')
      ->execute();
    $results = $result !== NULL ? $result->fetchCol() : [];

    $this->assertSame(['item_x', 'item_y'], $results);

    // Cleanup.
    $connectionManager->dropTempTable('test_idx', $tableName2);
    $connectionManager->deleteDatabase('test_idx');
    rmdir($tempDir);
  }

}
