<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Processor;

use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api\ServerInterface;
use Drupal\search_api_sqlite\Plugin\search_api\processor\Fts5Highlight;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the Fts5Highlight processor.
 */
#[CoversClass(Fts5Highlight::class)]
#[Group('search_api_sqlite')]
class Fts5HighlightTest extends UnitTestCase {

  /**
   * Tests that the processor only supports SQLite backend indexes.
   */
  public function testSupportsIndex(): void {
    // Index with SQLite backend.
    $sqlite_server = $this->createMock(ServerInterface::class);
    $sqlite_server->method('getBackendId')->willReturn('search_api_sqlite');

    $sqlite_index = $this->createMock(IndexInterface::class);
    $sqlite_index->method('getServerInstance')->willReturn($sqlite_server);

    $this->assertTrue(Fts5Highlight::supportsIndex($sqlite_index));

    // Index with different backend.
    $other_server = $this->createMock(ServerInterface::class);
    $other_server->method('getBackendId')->willReturn('search_api_db');

    $other_index = $this->createMock(IndexInterface::class);
    $other_index->method('getServerInstance')->willReturn($other_server);

    $this->assertFalse(Fts5Highlight::supportsIndex($other_index));

    // Index without server.
    $no_server_index = $this->createMock(IndexInterface::class);
    $no_server_index->method('getServerInstance')->willReturn(NULL);

    $this->assertFalse(Fts5Highlight::supportsIndex($no_server_index));
  }

  /**
   * Tests default configuration.
   */
  public function testDefaultConfiguration(): void {
    $processor = $this->createProcessor();

    $config = $processor->defaultConfiguration();

    $this->assertEquals('<mark>', $config['prefix']);
    $this->assertEquals('</mark>', $config['suffix']);
    $this->assertEquals(64, $config['excerpt_length']);
    $this->assertEquals([], $config['exclude_fields']);
  }

  /**
   * Tests that preprocessSearchQuery sets the option.
   */
  public function testPreprocessSearchQuerySetsOption(): void {
    $processor = $this->createProcessor([
      'prefix' => '<strong>',
      'suffix' => '</strong>',
      'excerpt_length' => 32,
      'exclude_fields' => ['body'],
    ]);

    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn('test query');

    // Expect setOption to be called with the configuration.
    $query->expects($this->once())
      ->method('setOption')
      ->with(
        'search_api_sqlite_highlight',
        $this->callback(fn($config): bool => $config['prefix'] === '<strong>'
          && $config['suffix'] === '</strong>'
          && $config['excerpt_length'] === 32
          && $config['exclude_fields'] === ['body'])
      );

    // Expect tag to skip Search API's Highlight processor.
    $query->expects($this->once())
      ->method('addTag')
      ->with('search_api_skip_processor_highlight');

    $processor->preprocessSearchQuery($query);
  }

  /**
   * Tests that preprocessSearchQuery skips when no keys.
   */
  public function testPreprocessSearchQuerySkipsWithoutKeys(): void {
    $processor = $this->createProcessor();

    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn(NULL);

    // setOption should not be called.
    $query->expects($this->never())
      ->method('setOption');

    $processor->preprocessSearchQuery($query);
  }

  /**
   * Creates a processor instance with given configuration.
   *
   * @param array<string, mixed> $configuration
   *   The processor configuration.
   *
   * @return \Drupal\search_api_sqlite\Plugin\search_api\processor\Fts5Highlight
   *   The processor instance.
   */
  private function createProcessor(array $configuration = []): Fts5Highlight {
    $index = $this->createMock(IndexInterface::class);

    $processor = new Fts5Highlight(
      $configuration,
      'search_api_sqlite_highlight',
      [
        'id' => 'search_api_sqlite_highlight',
        'label' => 'SQLite FTS5 Highlighting',
        'description' => 'Test',
        'stages' => ['preprocess_query' => 0],
      ]
    );
    $processor->setIndex($index);

    return $processor;
  }

}
